#pragma once
#include <mail/template_master/lib/db/queries/query_repository.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/db/operations/operation_traits.h>
#include <mail/template_master/lib/types/template/database_template.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/types/expected.h>

#include <ozo/shortcuts.h>

#include <vector>

namespace NTemplateMaster::NDatabase::Operations {

template<typename TQueryRepository>
class TMergeTemplatesOp {
private:
    using TSelectForUpdateSearchFlag = NTemplateMaster::NDatabase::NQuery::TSelectForUpdateSearchFlag;
    using TUpdateSearchFlag = NTemplateMaster::NDatabase::NQuery::TUpdateSearchFlag;
    using TSaveTemplate = NTemplateMaster::NDatabase::NQuery::TSaveTemplate;
public:
    TMergeTemplatesOp(
            TQueryRepository queryRepository,
            TDatabaseTemplatePtr templ,
            int32_t featuresLimit,
            TTemplatesStableSigns stableSigns)
        : QueryRepository(std::move(queryRepository))
        , Template(std::move(templ))
        , StableSigns(std::move(stableSigns))
        , FeaturesLimit(featuresLimit)
    {}

    template<typename TProvider>
    TExpected<bool> operator()(
            NTemplateMaster::TContextPtr context,
            TProvider&& provider,
            ozo::time_traits::duration requestTimeout,
            TYield yield)
    {
        using Adl::execute;
        using Adl::begin;
        using Adl::commit;
        using Adl::request;
        using Adl::rollback;

        boost::system::error_code ec;

        std::vector<typename TSelectForUpdateSearchFlag::result_type> selectForUpdateSearchFlagResult;
        TSelectForUpdateSearchFlag selectForUpdateSearchFlagParams;
        selectForUpdateSearchFlagParams.search = true;
        selectForUpdateSearchFlagParams.stable_sign_array = StableSigns;
        const auto selectForUpdateSearchFlagQ =
                QueryRepository.template make_query<TSelectForUpdateSearchFlag>(selectForUpdateSearchFlagParams);

        TUpdateSearchFlag updateSearchFlagParams;
        updateSearchFlagParams.search = false;
        updateSearchFlagParams.stable_sign_array = StableSigns;
        const auto updateSearchFlagsQ = QueryRepository.template make_query<TUpdateSearchFlag>(updateSearchFlagParams);

        std::vector<typename TSaveTemplate::result_type> saveResult;
        TSaveTemplate saveTemplateParams;
        saveTemplateParams.stable_sign = Template->GetStableSign();
        saveTemplateParams.features = {Template->GetFeatures().begin(), Template->GetFeatures().end()};
        saveTemplateParams.features_limit = FeaturesLimit;
        saveTemplateParams.parent_templates_stable_signs = StableSigns;
        saveTemplateParams.attributes = Template->GetJsonAttributesArray();
        saveTemplateParams.body = ozo::pg::jsonb(Template->Json());
        const auto saveQ = QueryRepository.template make_query<TSaveTemplate>(saveTemplateParams);

        auto transaction = begin(std::forward<TProvider>(provider), requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, "begin", transaction);
            return yamail::make_unexpected(ec);
        }

        request(transaction, selectForUpdateSearchFlagQ, requestTimeout,
                ozo::into(selectForUpdateSearchFlagResult), yield[ec]);
        if (ec) {
            LogOzoError(context, ec, selectForUpdateSearchFlagQ, transaction);
            return yamail::make_unexpected(ec);
        }
        if (selectForUpdateSearchFlagResult.size() != StableSigns.size()) {
            rollback(std::move(transaction), requestTimeout, yield[ec]);
            if (ec) {
                LogOzoError(context, ec, "rollback", transaction);
                return yamail::make_unexpected(ec);
            }
            return {false};
        }

        execute(transaction, updateSearchFlagsQ, requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, updateSearchFlagsQ, transaction);
            return yamail::make_unexpected(ec);
        }
        request(transaction, saveQ, requestTimeout, ozo::into(saveResult), yield[ec]);
        if (ec) {
            LogOzoError(context, ec, saveQ, transaction);
            return yamail::make_unexpected(ec);
        }
        if (saveResult.empty()) {
            TUpdateSearchFlag updateSearchFlagForExistsTemplateParams;
            updateSearchFlagForExistsTemplateParams.search = true;
            updateSearchFlagForExistsTemplateParams.stable_sign_array = {Template->GetStableSign()};
            const auto updateSearchFlagForExistsTemplateQ =
                    QueryRepository.template make_query<TUpdateSearchFlag>(updateSearchFlagForExistsTemplateParams);
            execute(transaction, updateSearchFlagForExistsTemplateQ, requestTimeout, yield[ec]);
            if (ec) {
                LogOzoError(context, ec, updateSearchFlagForExistsTemplateQ, transaction);
                return yamail::make_unexpected(ec);
            }
        }
        commit(std::move(transaction), requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, "commit", transaction);
            return yamail::make_unexpected(ec);
        }
        return {true};
    }
private:
    const TQueryRepository QueryRepository;
    const TDatabaseTemplatePtr Template;
    const TTemplatesStableSigns StableSigns;
    const int32_t FeaturesLimit;
};

}
