#pragma once
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/log.h>
#include <mail/template_master/lib/db/queries/query_repository.h>
#include <mail/template_master/lib/types/asio.h>

#include <ozo/request.h>
#include <ozo/execute.h>
#include <ozo/transaction.h>
#include <ozo/connection.h>

namespace NTemplateMaster::NDatabase::Operations {

using TQueryRepository = NTemplateMaster::NDatabase::NQuery::TQueryRepository;

template <typename Operation, typename Provider>
using TOperationReturnType = decltype(std::declval<Operation>()(
        std::declval<NTemplateMaster::TContextPtr>(),
        std::declval<Provider>(),
        std::declval<ozo::time_traits::duration>(),
        std::declval<TYield>()));

template<typename TConn>
inline decltype(auto) GetLogErrorMessage(TConn&& connection) {
    using ozo::get_error_context;
    return connection ? get_error_context(std::forward<TConn>(connection)) : std::string();
}

template<typename TConn>
inline decltype(auto) GetLogPqMessage(TConn&& connection) {
    using ozo::error_message;
    return connection ? error_message(std::forward<TConn>(connection)) : std::string_view();
}

template<typename TConn, typename TQuery>
void LogOzoError(
        NTemplateMaster::TContextPtr context,
        boost::system::error_code ec,
        TQuery&& query,
        TConn&& connection)
{
    using ozo::get_text;
    using ozo::to_const_char;
    LOGDOG_(context->GetLogger(), error,
            NTemplateMaster::NLog::error_code=ec,
            NTemplateMaster::NLog::query=to_const_char(get_text(std::forward<TQuery>(query))),
            NTemplateMaster::NLog::message=GetLogErrorMessage(std::forward<TConn>(connection)),
            NTemplateMaster::NLog::pq_message=GetLogPqMessage(std::forward<TConn>(connection)))
}

template<typename TConn>
void LogOzoError(
        NTemplateMaster::TContextPtr context,
        boost::system::error_code ec,
        const char* query,
        TConn&& connection)
{
    LOGDOG_(context->GetLogger(), error,
            NTemplateMaster::NLog::error_code=ec,
            NTemplateMaster::NLog::query=query,
            NTemplateMaster::NLog::message=GetLogErrorMessage(std::forward<TConn>(connection)),
            NTemplateMaster::NLog::pq_message=GetLogPqMessage(std::forward<TConn>(connection)))
}

namespace Adl {

template <typename T>
struct TRequestImpl {
    template <typename ...Ts>
    static auto Apply(Ts&& ...v) {
        return ozo::request(std::forward<Ts>(v)...);
    }
};

template <class T, class ...Ts>
auto request(T&& v, Ts&& ...vs) {
    return TRequestImpl<std::decay_t<T>>::Apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

template <typename T>
struct TExecuteImpl {
    template <typename ...Ts>
    static auto Apply(Ts&& ...v) {
        return ozo::execute(std::forward<Ts>(v)...);
    }
};

template <class T, class ...Ts>
auto execute(T&& v, Ts&& ...vs) {
    return TExecuteImpl<std::decay_t<T>>::Apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

template <typename T>
struct TBeginImpl {
    template <typename ...Ts>
    static auto Apply(Ts&& ...v) {
        return ozo::begin(std::forward<Ts>(v)...);
    }
};

template <class T, class ...Ts>
auto begin(T&& v, Ts&& ...vs) {
    return TBeginImpl<std::decay_t<T>>::Apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

template <typename T>
struct TCommitImpl {
    template <typename ...Ts>
    static auto Apply(Ts&& ...v) {
        return ozo::commit(std::forward<Ts>(v)...);
    }
};

template <class T, class ...Ts>
auto commit(T&& v, Ts&& ...vs) {
    return TCommitImpl<std::decay_t<T>>::Apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}

template <typename T>
struct TRollbackImpl {
    template <typename ...Ts>
    static auto Apply(Ts&& ...v) {
        return ozo::rollback(std::forward<Ts>(v)...);
    }
};

template <class T, class ...Ts>
auto rollback(T&& v, Ts&& ...vs) {
    return TRollbackImpl<std::decay_t<T>>::Apply(std::forward<T>(v), std::forward<Ts>(vs)...);
}
}

}
