#pragma once
#include <mail/template_master/lib/db/queries/query_repository.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/db/operations/operation_traits.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/types/expected.h>

#include <ozo/execute.h>
#include <ozo/pg/types/jsonb.h>

#include <vector>

namespace NTemplateMaster::NDatabase::Operations {

template<typename TQueryRepository>
class TSaveTemplateOp {
private:
    using TSaveTemplate = NTemplateMaster::NDatabase::NQuery::TSaveTemplate;
public:
    TSaveTemplateOp(
            TQueryRepository queryRepository,
            TDatabaseTemplatePtr templ,
            int32_t featuresLimit,
            TTemplatesStableSigns parentIds)
        : QueryRepository(std::move(queryRepository))
        , Template(std::move(templ))
        , FeaturesLimit(featuresLimit)
        , ParentTemplatesIds(std::move(parentIds))
    {}

    template<typename TProvider>
    TExpected<void> operator()(
            NTemplateMaster::TContextPtr context,
            TProvider&& provider,
            ozo::time_traits::duration requestTimeout,
            TYield yield) {
        using Adl::execute;
        using Adl::begin;
        using Adl::commit;

        boost::system::error_code ec;
        TSaveTemplate saveTemplateParams;
        saveTemplateParams.stable_sign = Template->GetStableSign();
        const auto& features = Template->GetFeatures();
        saveTemplateParams.features = {features.begin(), features.end()};
        saveTemplateParams.features_limit = FeaturesLimit;
        saveTemplateParams.parent_templates_stable_signs = ParentTemplatesIds;
        saveTemplateParams.attributes = Template->GetJsonAttributesArray();
        saveTemplateParams.body = ozo::pg::jsonb(Template->Json());
        const auto saveQ = QueryRepository.template make_query<TSaveTemplate>(saveTemplateParams);

        auto transaction = begin(std::forward<TProvider>(provider), requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, "begin", transaction);
            return yamail::make_unexpected(ec);
        }
        execute(transaction, saveQ, requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, saveQ, transaction);
            return yamail::make_unexpected(ec);
        }
        commit(std::move(transaction), requestTimeout, yield[ec]);
        if (ec) {
            LogOzoError(context, ec, "commit", transaction);
            return yamail::make_unexpected(ec);
        }
        return {};
    }
private:
    const TQueryRepository QueryRepository;
    const TDatabaseTemplatePtr Template;
    const int32_t FeaturesLimit;
    const TTemplatesStableSigns ParentTemplatesIds;
};

}
