#pragma once
#include <mail/template_master/lib/http/handler.h>
#include <mail/template_master/lib/http/utils.h>
#include <mail/template_master/lib/http/types.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/template_master/handlers/detemple_handler.h>
#include <mail/template_master/lib/unistat/unistat.h>

#include <mail/ymod_webserver/include/ymod_webserver/methods.h>
#include <mail/ymod_webserver/include/ymod_webserver/server.h>
#include <mail/yplatform/include/yplatform/find.h>

#include <string>

namespace NTemplateMaster::NHttp {

class TDetemple : public IHandler {
private:
    using TRequest = NTemplateMaster::THttpDetempleRequest;
    using TDetempleHandler = NTemplateMaster::NHandlers::TDetempleHandler<TContentProcessor, THttpRouteRequest, TMessagePtr>;
public:
    TDetemple(TReactorPtr reactor, TContentProcessorPtr contentProcessor)
        : Reactor(std::move(reactor))
        , ContentProcessor(std::move(contentProcessor))
    {}

    THttpStreamPtr Init(THttpStreamPtr stream) const override {
        return THttpStreamPtr(stream.get(), CreateUnistatCallback(stream, Paths()));
    }

    THttpPaths Paths() const override {
        return {"/force_detemple"};
    }

    EHttpMethod Method() const override {
        return EHttpMethod::mth_post;
    }

    bool NeedTvm() const override {
        return false;
    }

    void Execute(THttpStreamPtr stream) override {
        try {
            auto request = DeserializeRequest(stream);
            auto context = boost::make_shared<TContext>(stream->request()->ctx(), GetRequestId(stream), request.QueueId);
            ExecuteImpl(context, stream, request);
        } catch (const std::runtime_error& e) {
            LOGDOG_(NHandlers::GetLogger(stream->request()->ctx(), GetRequestId(stream)), error, NTemplateMaster::NLog::exception=e)
            stream->set_code(ymod_webserver::codes::bad_request);
        }
    }
private:
    TRequest DeserializeRequest(THttpStreamPtr stream) {
        TRequest req;
        const std::string body(stream->request()->raw_body.begin(), stream->request()->raw_body.end());
        yamail::data::deserialization::fromJson(body, req);
        return req;
    }

    void ExecuteImpl(TContextPtr context, THttpStreamPtr stream, const TRequest& req) {
        auto templateMaster = yplatform::find<ITemplateMaster, std::shared_ptr>("template_master");
        const auto msg = std::make_shared<TMessage>(*ContentProcessor, req);

        auto detempleHandler = std::make_shared<TDetempleHandler>(context, msg, templateMaster, req.DbHints);
        auto coro = NTemplateMaster::NUnistat::Wrap([=](TYield yield) mutable {
            const auto result = (*detempleHandler)(yield);
            SetOperationResult(result, stream);
        }, "api_/force_detemple_time");
        boost::asio::spawn(*Reactor->io(), std::move(coro), NTemplateMaster::NUtils::kCoroutineAttributes);
    }

private:
    TReactorPtr Reactor;
    TContentProcessorPtr ContentProcessor;
};
}
