#pragma once
#include <mail/template_master/lib/http/handler.h>
#include <mail/template_master/lib/http/utils.h>
#include <mail/template_master/lib/http/types.h>
#include <mail/template_master/lib/template_master/handlers/route_template_handler.h>
#include <mail/template_master/lib/types/context.h>

#include <mail/ymod_webserver/include/ymod_webserver/methods.h>
#include <mail/ymod_webserver/include/ymod_webserver/server.h>
#include <mail/yplatform/include/yplatform/find.h>

#include <string>

namespace NTemplateMaster::NHttp {

class TRoute : public IHandler {
private:
    using TRequest = NTemplateMaster::THttpRouteRequest;
    using TRouteHandler = NTemplateMaster::NHandlers::TRouteTemplateHandler<TMessagePtr>;
public:
    TRoute(TReactorPtr reactor, TContentProcessorPtr contentProcessor)
        : Reactor(std::move(reactor))
        , ContentProcessor(std::move(contentProcessor))
    {}

    THttpStreamPtr Init(THttpStreamPtr stream) const override {
        return THttpStreamPtr(stream.get(), CreateUnistatCallback(stream, Paths()));
    }

    THttpPaths Paths() const override {
        return {"/route"};
    }

    EHttpMethod Method() const override {
        return EHttpMethod::mth_post;
    }

    bool NeedTvm() const override {
        return false;
    }

    void Execute(THttpStreamPtr stream) override {
        try {
            auto body = boost::copy_range<std::string>(stream->request()->raw_body);
            auto request = DeserializeRequest(body);
            auto context = boost::make_shared<TContext>(stream->request()->ctx(), GetRequestId(stream), request.QueueId);
            NHandlers::LogDetempleRequest(context, body);
            ExecuteImpl(context, stream, std::move(body), request);
        } catch (const std::runtime_error& e) {
            LOGDOG_(NHandlers::GetLogger(stream->request()->ctx(), GetRequestId(stream)), error, NTemplateMaster::NLog::exception=e)
            stream->set_code(ymod_webserver::codes::bad_request);
        }
    }
private:
    TRequest DeserializeRequest(const std::string& body) {
        TRequest req;
        yamail::data::deserialization::fromJson(body, req);
        return req;
    }

    void ExecuteImpl(TContextPtr context, THttpStreamPtr stream, std::string body, const TRequest& req) {
        auto templateMaster = yplatform::find<ITemplateMaster, std::shared_ptr>("template_master");
        auto router = yplatform::find<NTemplateMaster::NRouter::IRouter, std::shared_ptr>("router");
        const auto msg = std::make_shared<TMessage>(*ContentProcessor, req);
        auto routeHandler = std::make_shared<TRouteHandler>(context, std::move(body), msg, templateMaster, router);
        auto coro = NTemplateMaster::NUnistat::Wrap([=](TYield yield) mutable {
            const auto result = (*routeHandler)(yield);
            NTemplateMaster::NUnistat::CollectOperationStatus(result);
            SetOperationResult(result, stream);
        }, "api_/route_time");
        boost::asio::spawn(*Reactor->io(), std::move(coro), NTemplateMaster::NUtils::kCoroutineAttributes);
    }

private:
    TReactorPtr Reactor;
    TContentProcessorPtr ContentProcessor;
};
}
