#pragma once
#include <mail/template_master/lib/types/time.h>
#include <mail/yplatform/include/yplatform/time_traits.h>

#include <util/system/spinlock.h>
#include <util/system/guard.h>

#include <string>

namespace NTemplateMaster::NRouter {

class INode {
public:
    virtual ~INode() = default;

    virtual const std::string& GetAddress() const = 0;

    virtual bool IsBanned() const = 0;

    virtual TDuration Ban(TDuration baseDuration) = 0;

    virtual int ResetBan() = 0;

    static TDuration CalcBanDuration(TDuration base, int num) {
        return base * (num + 1);
    }
};

template<typename TNow = decltype(&TClock::now)>
class TNode : public INode {
public:
    TNode(std::string address, TNow now = &TClock::now)
        : Address(std::move(address))
        , BanNumber(0)
        , BannedUntil(TTimePoint::min())
        , Now(now)
    {}

    const std::string& GetAddress() const override {
        return Address;
    }

    bool IsBanned() const override {
        auto guard = Guard(Lock);
        return BannedUntil >= Now();
    }

    TDuration Ban(TDuration baseDuration) override {
        auto now = Now();
        auto duration = TDuration::zero();
        auto guard = Guard(Lock);
        if (now > BannedUntil) {
            duration = CalcBanDuration(baseDuration, BanNumber++);
            BannedUntil = now + duration;
        }
        return duration;
    }

    int ResetBan() override {
        auto guard = Guard(Lock);
        int banNumber = BanNumber;
        BanNumber = 0;
        BannedUntil = TTimePoint::min();
        return banNumber;
    }
private:
    std::string Address;
    int BanNumber;
    TTimePoint BannedUntil;
    mutable TAdaptiveLock Lock;
    TNow Now;
};

using TNodePtr = std::shared_ptr<INode>;
}
