#pragma once
#include <mail/template_master/lib/router/node_manager.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/utils/errors.h>
#include <mail/template_master/lib/unistat/unistat.h>
#include <mail/template_master/lib/http/utils.h>
#include <mail/template_master/lib/types/http.h>
#include <mail/template_master/lib/types/time.h>
#include <mail/template_master/lib/types/expected.h>
#include <mail/template_master/lib/types/asio.h>

#include <mail/ymod_httpclient/include/ymod_httpclient/client.h>
#include <mail/butil/include/butil/http/headers.h>

namespace NTemplateMaster::NRouter {

template<typename TNodeGenerator>
class TRequestWithRetries {
public:
    TRequestWithRetries(
            THttpOptions opts,
            size_t maxRetries,
            TNodeGenerator nodeGenerator,
            std::string schema,
            size_t port,
            std::string body,
            yplatform::time_traits::duration banDuration,
            std::shared_ptr<yhttp::simple_call> httpClient)
        : NodeGenerator(std::move(nodeGenerator))
        , Options(std::move(opts))
        , MaxRetries(maxRetries)
        , CurrentRetry(0)
        , Schema(std::move(schema))
        , Port(std::to_string(port))
        , Body(std::move(body))
        , BanDuration(banDuration)
        , HttpClient(std::move(httpClient))
    {}

    TExpected<THttpResponse> Run(TContextPtr context, TYield yield) {
        boost::system::error_code ec;
        auto node = NodeGenerator();
        auto req = MakeRequest(context, node, Body);
        const auto response = HttpClient->async_run(context, std::move(req), Options, yield[ec]);
        NTemplateMaster::NUnistat::CollectHttpRequestResult(ec, response.status, "http_request_try");
        if (ec || response.status / 100 == 5) {
            node->Ban(BanDuration);
            if (ec) {
                const std::string message = "router, fail request to: " + node->GetAddress() + " retry: "
                        + std::to_string(CurrentRetry);
                LOGDOG_(context->GetLogger(), error,
                        NTemplateMaster::NLog::error_code=ec,
                        NTemplateMaster::NLog::message=message)
            }
        } else {
            node->ResetBan();
            return MakeExpected(response);
        }
        if (CurrentRetry >= MaxRetries) {
            return ec ? yamail::make_unexpected(ec) : MakeExpected(response);
        }
        CurrentRetry++;
        return Run(context, yield);
    }

private:
    THttpRequest MakeRequest(const TContextPtr& context, TNodePtr node, std::string body) const {
        const std::string url = Schema + "://" + node->GetAddress() + ":" + Port + "/force_detemple";
        LOGDOG_(context->GetLogger(), notice,
                NTemplateMaster::NLog::type="Route",
                NTemplateMaster::NLog::url=url)
        http::headers headers;
        headers.add(NTemplateMaster::NHttp::xRequestIdKey, context->GetRequestId());
        return THttpRequest::POST(url, headers.format(), std::move(body));
    }

private:
    TNodeGenerator NodeGenerator;
    const THttpOptions Options;
    const size_t MaxRetries;
    size_t CurrentRetry;
    const std::string Schema;
    const std::string Port;
    const std::string Body;
    const TDuration BanDuration;
    const std::shared_ptr<yhttp::simple_call> HttpClient;
};

}
