#pragma once
#include <mail/template_master/lib/router/node_manager.h>
#include <mail/template_master/lib/router/router_module.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/utils/errors.h>
#include <mail/template_master/lib/router/config.h>
#include <mail/template_master/lib/router/nanny_meta.h>
#include <mail/template_master/lib/http/utils.h>
#include <mail/template_master/lib/router/retry.h>

#include <util/digest/numeric.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/consistent_hashing/consistent_hashing.h>
#include <mail/ymod_http_watcher/watcher.h>

#include <mail/ymod_httpclient/include/ymod_httpclient/client.h>
#include <mail/butil/include/butil/http/headers.h>

#include <mail/yreflection/include/yamail/data/deserialization/yajl.h>
#include <yplatform/coroutine.h>
#include <boost/algorithm/string.hpp>

namespace NTemplateMaster::NRouter {

class TRouter : public IRouter {
public:
    void init(const yplatform::ptree& config) {
        Config = yamail::data::deserialization::fromPtree<TConfig>(config);
        Options.timeouts.connect = Config.ConnectTimeout;
        Options.timeouts.total = Config.TotalTimeout;
        Options.reuse_connection = Config.ReuseConnection;

        if (!Config.NannyDumpPath.empty()) {
            NannyMeta.Init(Config.NannyDumpPath);
        }
        auto watcher = yplatform::find<NYmodHttpWatcher::TWatcher>("nanny_watcher");
        watcher->SetHandler([this](const yhttp::response& response) {
            boost::property_tree::ptree tree;
            std::stringstream bodystream(response.body);
            boost::property_tree::read_json(bodystream, tree);
            std::vector<std::string> newHosts;
            for (auto& item : tree) {
                auto& instance = item.second;
                std::string dc;
                for (auto& tagItem : instance.get_child("itags")) {
                    std::string tag(tagItem.second.data());
                    if (tag.starts_with("a_dc_")) {
                        dc = tag.substr(5);
                        break;
                    }
                }
                if (!dc.empty() && dc == GetDC().value()) {
                    newHosts.push_back(std::move(dc));
                }
            }
            UpdateHostsIfNeeded(std::move(newHosts));
        });
        watcher->SetErrorHandler([this](const auto& ec) {
            YLOG_L(error) << "Nanny API request failed: " << ec.message();
        });
    }

    TOptional<std::string> GetDC() const noexcept override {
        return NannyMeta.GetDC();
    }

    TExpected<THttpResponse> SendRequest(
        TContextPtr context,
        std::string body,
        TTemplateFeaturesSet features,
        TYield yield) noexcept override
    {
        const auto nodeGenerator = [=]() -> TNodePtr {
            TNodeManager::TNodes available_nodes = GetNodeManager()->GetAvailableNodes();
            if (available_nodes.empty()) {
                return std::make_shared<TNode<>>("localhost");
            }
            // minhash is better: https://paste.yandex-team.ru/1140229
            TTemplateFeature min_feature = *std::min_element(features.begin(), features.end());
            ui64 key = IntHash<ui64>(min_feature); // scramble, for uniform distribution
            uint node_id = ConsistentHashing(key, available_nodes.size());
            return available_nodes.at(node_id);
        };
        auto httpClient = yplatform::find<yhttp::simple_call, std::shared_ptr>("http_client");
        TRequestWithRetries requestWithRetries(Options, Config.Retries, nodeGenerator, Config.Schema,
                Config.Port, std::move(body), Config.BanDuration, httpClient);
        return requestWithRetries.Run(context, yield);
    }

private:
    void UpdateHostsIfNeeded(std::vector<std::string> hosts) noexcept {
        if (CurrentHosts != hosts) {
            CurrentHosts = std::move(hosts);
            if (!CurrentHosts.empty()) {
                TNodeManager::TNodes nodes;
                for (const auto& host : CurrentHosts) {
                    nodes.emplace_back(std::make_shared<TNode<>>(host));
                }
                auto nodeManager = std::make_shared<TNodeManager>(std::move(nodes));
                UpdateNodeManager(nodeManager);
            }
        }
    }



private:
    THttpOptions Options;
    TConfig Config;
    TNannyMeta NannyMeta;
    std::vector<std::string> CurrentHosts;
};

}
