#pragma once
#include <mail/template_master/lib/types/token/hash.h>
#include <mail/template_master/lib/types/template/stable_sign.h>
#include <mail/template_master/lib/types/template/template_sign.h>
#include <mail/template_master/lib/template_master/content_processor/email_attributes.h>
#include <mail/template_master/lib/types/operation_request/email_http_request.h>

#include <boost/algorithm/string.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/scheme/scheme.h>

#include <type_traits>
#include <algorithm>
#include <unordered_set>
#include <iomanip>

namespace NTemplateMaster {

using THighfreqHashes = std::shared_ptr<std::unordered_set<TTemplateFeature>>;

class TShingerPrint {
public:
    TShingerPrint(size_t maxEqualTokens)
        : MaxEqualTokens(maxEqualTokens)
    {}

    /*
     * It's black magic
     */
    auto CreateShinger(const std::string& s) const noexcept {
        int64_t result = 0;
        MD5 md5;
        auto hash = static_cast<std::string>(md5.CalcRaw(s));
        for (size_t i = 0; i != 3; ++i) {
            result = result | static_cast<unsigned char>(hash[i]);
            result = result << 8;
        }
        const auto c = static_cast<unsigned char>(hash[3]);
        result = result | c & 0xf0;
        result = result >> 4;
        return static_cast<int32_t>(result);
    }

    template<typename TRequest>
    auto Tokenize(TRequest request) const noexcept {
        const auto s = request.Html;

        TStringTokenSequence result;
        size_t prevEnd = 0;
        for (size_t index = 0; index != s.size(); ++index) {
            char c = s[index];
            if ('<' == c && prevEnd != index) {
                result.emplace_back(s.substr(prevEnd, index - prevEnd));
                prevEnd = index;
            } else if ('>' == c) {
                result.emplace_back(s.substr(prevEnd, index - prevEnd + 1));
                prevEnd = index + 1;
            }
        }
        if (prevEnd != s.length()) {
            result.emplace_back(s.substr(prevEnd, s.size() - prevEnd));
        }

        TStringTokenSequence newTokens;
        std::unordered_map<TStringToken, size_t> tokenFreq;
        for (size_t i = 0; i < result.size(); i++) {
            ++tokenFreq[result[i]];
            if (i + 1 < result.size() && tokenFreq[result[i]] > MaxEqualTokens) {
                result[i + 1] = result[i] + result[i + 1];
                continue;
            }
            newTokens.emplace_back(std::move(result[i]));
        }
        return newTokens;
    }

    template<typename TRequest>
    TEmailAttributes ExtractAttributes(TRequest request) const noexcept {
        return {request.From, request.Subject, request.QueueId, request.Uids};
    }

    auto CalculateDigest(const TStringTokenSequence& tokens) const noexcept {
        TTemplateFeaturesSet digest;
        for (auto&& token : tokens) {
            if (!token.IsSentinel()) {
                digest.emplace(CreateShinger(token.GetValue()));
            }
        }
        return digest;
    }

    auto CalculateStableSign(TStringTokenSequence tokens) const noexcept {
        const auto s = boost::algorithm::join(tokens | boost::adaptors::transformed(
                [](auto&& token) {
                    return token.GetValue();
                }), "");
        int64_t result = 0;
        MD5 md5;
        const auto hash = static_cast<std::string>(md5.CalcRaw(s));
        std::stringstream ss;
        ss << std::hex << std::setfill('0');
        for (auto&& c : hash) {
            ss << std::setw(2) << static_cast<i32>(static_cast<unsigned char>(c));
        }
        std::stringstream digest;
        digest << std::hex << ss.str().substr(0, 15);
        digest >> result;
        return result;
    }

    auto ToJson(TStringTokenSequence tokens) const {
        NJson::TJsonValue result = NJson::JSON_ARRAY;

        NJson::TJsonValue arr = NJson::JSON_ARRAY;
        for (auto&& token : tokens) {
            if (token.IsSentinel() && arr.GetArray().size()) {
                result.AppendValue(std::move(arr));
            } else if (!token.IsSentinel()) {
                arr.AppendValue(NJson::TJsonValue(token.GetValue()));
            }
        }
        if (arr.GetArray().size()) {
            result.AppendValue(std::move(arr));
        }
        NJsonWriter::TBuf w(NJsonWriter::EHtmlEscapeMode::HEM_UNSAFE);
        w.WriteJsonValue(&result);
        return static_cast<std::string>(w.Str());
    }

    auto FromJson(const std::string& jsonChunks) {
        NSc::TValue chunks = NSc::TValue::FromJson(jsonChunks);
        TStringTokenSequence templateTokens;
        templateTokens.emplace_back();
        for (auto&& tokens : chunks.GetArray()) {
            if (!tokens.GetArray().empty()) {
                for (auto&& token : tokens.GetArray()) {
                    templateTokens.emplace_back(token.GetString().Data());
                }
                templateTokens.emplace_back();
            }
        }
        return templateTokens;
    }
private:
    const size_t MaxEqualTokens;
};

using TShingerPrintPtr = std::shared_ptr<TShingerPrint>;
}
