#pragma once
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/template/message.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/types/template/database_template.h>

#include <boost/range/adaptor/transformed.hpp>
#include <boost/algorithm/string/join.hpp>

namespace NTemplateMaster::NHandlers {

template<typename TContentProcessor, typename TContent, class TTokenType = ContentProcessorTokenType<TContentProcessor, TContent>>
inline void LogDiffWithTemplate(
        TContextPtr context,
        TStableTemplatePtr<TContentProcessor, TContent> templ,
        const TMatchResult<TTokenType>& matchResult)
{
    std::stringstream res;
    NUtils::TFormatter<TTokenType> fmt;
    NUtils::PrintChunks(res, fmt, matchResult.GetDiff());
    LOGDOG_(context->GetDebugLogger(), debug,
            NTemplateMaster::NLog::type="DiffWithDBTemplate",
            NTemplateMaster::NLog::stable_sign=templ->GetStableSign(),
            NTemplateMaster::NLog::full_match=matchResult.IsFullMatch(),
            NTemplateMaster::NLog::diff=res.str())
}

template<typename TDetempleObjectPtr>
inline void LogMessageDigest(TContextPtr context, TDetempleObjectPtr msg) {
    LOGDOG_(context->GetLogger(), notice,
            NTemplateMaster::NLog::type="Email",
            NTemplateMaster::NLog::digest=msg->GetFeatures())
}

template<typename TTemplates>
inline void LogFoundInDbTemplates(TContextPtr context, TTemplates templates) {
    std::vector<NTemplateMaster::TTemplateStableSign> templatesStableSigns;
    std::transform(templates.begin(), templates.end(), std::back_inserter(templatesStableSigns), [](auto&& t) {
        return t->GetStableSign();
    });
    LOGDOG_(context->GetLogger(), notice,
            NTemplateMaster::NLog::type="FoundInDb",
            NTemplateMaster::NLog::stable_signs=templatesStableSigns)
}

inline void LogDetempleRequest(TContextPtr context, const std::string& body) {
    LOGDOG_(context->GetDebugLogger(), debug,
            NTemplateMaster::NLog::type="Request",
            NTemplateMaster::NLog::body=body)
}

template<typename TTokenType, typename TAttributes>
inline void LogDetempleResult(TContextPtr context, TDetempleResult<TTokenType, TAttributes> result) {
    LOGDOG_(context->GetDebugLogger(), debug,
            NTemplateMaster::NLog::type="Response",
            NTemplateMaster::NLog::detemple_result=yamail::data::serialization::toJson(result).str())
}

inline auto GetLogger(ymod_webserver::context_ptr ctx, const std::string& requestId) {
    return NTemplateMaster::NLog::GetLogger(NTemplateMaster::NLog::templateMasterLogKey, ctx.get()->uniq_id(), requestId);
}

}
