#pragma once
#include <mail/template_master/lib/db/database_module.h>
#include <mail/template_master/lib/http/types.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/router/router_module.h>
#include <mail/template_master/lib/template_master/template_master.h>
#include <mail/template_master/lib/template_master/content_processor/traits.h>
#include <mail/template_master/lib/template_master/handlers/log_utils.h>
#include <mail/template_master/lib/unistat/unistat.h>

#include <mail/yreflection/include/yamail/data/deserialization/yajl.h>

namespace NTemplateMaster::NHandlers {

template<typename TDetempleObjectPtr>
class TRouteTemplateHandler {
public:
    using TContentProcessor = NTemplateMaster::NHttp::TContentProcessor;
    using TContent = NTemplateMaster::THttpRouteRequest;
    static_assert(ContentProcessorConcept<TContentProcessor, TContent>, "must satisfy ContentProcessor concept");
    using TAttributes = ContentProcessorAttributesType<TContentProcessor, TContent>;
    using TTokenType = ContentProcessorTokenType<TContentProcessor, TContent>;
    using TTemplateMasterPtr = NTemplateMaster::TTemplateMasterPtr<TContentProcessor, TContent, TDetempleObjectPtr>;
    using TRouterPtr = NTemplateMaster::NRouter::TRouterPtr;

    TRouteTemplateHandler(
            NTemplateMaster::TContextPtr context,
            std::string body,
            TDetempleObjectPtr message,
            TTemplateMasterPtr templateMaster,
            TRouterPtr router)
        : Context(std::move(context))
        , Body(std::move(body))
        , Message(std::move(message))
        , ContentProcessor(Message->GetContentProcessor())
        , TemplateMaster(std::move(templateMaster))
        , Router(std::move(router))
    {}

    auto CreateStableTemplates(TDatabaseTemplates dbTemplates) {
        TStableTemplates<TContentProcessor, TContent> stableTemplates;
        for (auto&& templ : dbTemplates) {
            stableTemplates.emplace_back(
                    templ->template CreateStableTemplate<TContentProcessor, TContent>(ContentProcessor));
        }
        return stableTemplates;
    }

    decltype(auto) MatchDbTemplate(TStableTemplatePtr<TContentProcessor, TContent> stableTemplate) {
        return NTemplateMaster::NUnistat::WrapWithLog([msg = Message, templ=std::move(stableTemplate)]() {
            return NUtils::MatchDbTemplate(msg->GetTokens(), templ->GetTokens());
        }, Context, "application_detemple_MatchDbTemplate")();
    }

    auto operator()(TYield yield) -> TDetempleResult<TTokenType, TAttributes> {
        LogMessageDigest(Context, Message);
        if (Message->Size() == 0) {
            return {};
        }
        // Top N via simlar
        auto dbTemplates = TemplateMaster->FindSimilarTemplates(Context, Message, yield);
        if (!dbTemplates) {
            const auto res = TDetempleResult<TTokenType, TAttributes>(EDetempleStatus::Error);
            LogDetempleResult(Context, res);
            return res;
        }
        LogFoundInDbTemplates(Context, dbTemplates.value());
        const auto stableTemplates = CreateStableTemplates(dbTemplates.value());

        TemplateMaster->MergeTemplates(Context, stableTemplates);
        for (auto&& stableTemplate : stableTemplates) {
            const auto deltaOpt = MatchDbTemplate(stableTemplate);
            if (deltaOpt.has_value()) {
                const auto res = TDetempleResult<TTokenType, TAttributes>
                    (deltaOpt.value(), stableTemplate->GetStableSign());
                LogDetempleResult(Context, res);
                return res;
            }
        }

        auto request = MakeRequest(Message->GetContent(), dbTemplates.value());
        const auto response = Router->SendRequest(Context, std::move(request), Message->GetFeatures(), yield);
        if (response) {
            return yamail::data::deserialization::fromJson
                    <TDetempleResult<TTokenType, TAttributes>>(response.value().body);
        }
        const auto res = TDetempleResult<TTokenType, TAttributes>(EDetempleStatus::Error);
        LogDetempleResult(Context, res);
        return res;
    }
private:
    auto MakeDbHints(const TDatabaseTemplates& dbTemplates) {
        TDbHints dbHints;
        dbHints.reserve(dbTemplates.size());
        // We are not limiting the number of templates to forward because there is already the
        // pretty restrictive `template_master.templates_limit_find_in_db` option in config
        boost::range::transform(dbTemplates, std::back_inserter(dbHints), [](const auto& templ) -> TDbHint {
            return {templ->GetStableSign(),
                    {templ->GetFeatures().begin(), templ->GetFeatures().end()},
                    templ->Json(),
                    templ->GetJsonAttributesArray().raw_string()};
        });
        return dbHints;
    }

    auto MakeRequest(const TContent& routeRequest, const TDatabaseTemplates& dbTemplates) {
        THttpDetempleRequest request{routeRequest, MakeDbHints(dbTemplates)};
        return yamail::data::serialization::toJson(request);
    }

    NTemplateMaster::TContextPtr Context;
    std::string Body;
    const TDetempleObjectPtr Message;
    const TContentProcessor ContentProcessor;
    TTemplateMasterPtr TemplateMaster;
    TRouterPtr Router;
};
}
