#pragma once
#include <mail/template_master/lib/db/database_module.h>
#include <mail/template_master/lib/types/template/template_sign.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/time.h>
#include <mail/template_master/lib/types/asio.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/utils/errors.h>

#include <mail/yplatform/include/yplatform/find.h>

#include <memory>
#include <unordered_set>
#include <atomic>

namespace NTemplateMaster {

struct TFeaturesCache {
    TFeaturesCache(TTemplateFeaturesSet features, bool init)
        : Features(std::move(features))
        , LastUpdateTs(TClock::now())
        , Init(init)
    {}

    const TTemplateFeaturesSet Features;
    const TTimePoint LastUpdateTs;
    bool Init;
};

using TFeaturesCachePtr = std::shared_ptr<TFeaturesCache>;

class TFilterFeatures : public std::enable_shared_from_this<TFilterFeatures> {
private:
    auto CreateContext() const noexcept {
        auto ctx = boost::make_shared<ymod_webserver::context>();
        auto context = boost::make_shared<TContext>(ctx, "TFilterFeatures");
        return context;
    }
public:
    TFilterFeatures(
            TIOContext* io,
            yplatform::time_traits::milliseconds featuresUpdateInterval,
            size_t featuresLimit)
        : Strand(*io)
        , TimerUpdateFeatures(Strand.get_io_context())
        , FeaturesUpdateInterval(featuresUpdateInterval)
        , FeaturesCache(std::make_shared<TFeaturesCache>(std::unordered_set<TTemplateFeature>{}, false))
        , FeaturesLimit(featuresLimit)
    {}


    // Lowest $FeaturesLimit features
    // blacklisted by (SELECT feature FROM template_features WHERE used = false)
    TExpected<TTemplateFeaturesSet> Filter(TContextPtr context, TTemplateFeaturesSet features) noexcept {
        auto featuresCache = std::atomic_load_explicit(&FeaturesCache, std::memory_order_acquire);
        if (!featuresCache->Init) {
            const auto ec = NErrors::MakeErrorCode(NErrors::EError::CacheNotInited);
            return yamail::make_unexpected(ec);
        }
        TTemplateFeaturesVector result;
        for (auto&& feature : features) {
            if (!featuresCache->Features.count(feature)) {
                result.emplace_back(feature);
            }
        }
        std::sort(result.begin(), result.end());
        result.resize(std::min(result.size(), FeaturesLimit));
        TTemplateFeaturesSet filterFeatures(result.begin(), result.end());
        LOGDOG_(context->GetLogger(), notice,
                NTemplateMaster::NLog::type="TFilterFeatures",
                NTemplateMaster::NLog::digest=filterFeatures);
        return filterFeatures;
    }

    void Start() {
        boost::asio::spawn(Strand, [self=shared_from_this()](TYield yield) mutable {
            boost::system::error_code ec;
            while (ec != boost::asio::error::operation_aborted) {
                self->UpdateCache(yield);
                self->TimerUpdateFeatures.expires_from_now(boost::posix_time::milliseconds(self->FeaturesUpdateInterval.count()));
                self->TimerUpdateFeatures.async_wait(yield[ec]);
            }
        }, NUtils::kCoroutineAttributes);
    }

    void Stop() {
        TimerUpdateFeatures.cancel();
    }
private:
    TExpected<void> UpdateCache(TYield yield) {
        auto context = CreateContext();
        auto db = yplatform::find<NDatabase::IDatabase>("database");
        const auto featuresExpected = db->GetUnusedFeatures(context, yield);
        if (!featuresExpected) {
            LOGDOG_(context->GetLogger(), error, NTemplateMaster::NLog::error_code=featuresExpected.error())
            return yamail::make_unexpected(featuresExpected.error());
        }
        auto features = featuresExpected.value();
        LOGDOG_(context->GetLogger(), notice,
                NTemplateMaster::NLog::type="TFilterFeatures",
                NTemplateMaster::NLog::message="Got " + std::to_string(features.size()) + " features")
        auto featuresCache = std::make_shared<TFeaturesCache>(TTemplateFeaturesSet{features.begin(), features.end()}, true);
        std::atomic_exchange_explicit(&FeaturesCache, featuresCache, std::memory_order_acq_rel);
        return {};
    }
private:
    TStrand Strand;
    TDeadlineTimer TimerUpdateFeatures;
    yplatform::time_traits::milliseconds FeaturesUpdateInterval;
    TFeaturesCachePtr FeaturesCache;
    size_t FeaturesLimit;
};

using TFilterFeaturesPtr = std::shared_ptr<TFilterFeatures>;

}
