#pragma once
#include <mail/template_master/lib/db/database_module.h>
#include <mail/template_master/lib/types/match/match_result.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/utils/utils.h>

#include <util/random/random.h>

#include <memory>

namespace NTemplateMaster {

template<typename TContentProcessor, typename TContent>
class TMergeSimilarTemplates {
public:
    TMergeSimilarTemplates(
            NTemplateMaster::NDatabase::TDatabasePtr db,
            TContentProcessor contentProcessor,
            double jacquardMinValueForMerge,
            double commonPartPercentMinValueForMerge,
            const int32_t featuresLimit)
        : Db(std::move(db))
        , ContentProcessor(std::move(contentProcessor))
        , JacquardMinValueForMerge(jacquardMinValueForMerge)
        , CommonPartPercentMinValueForMerge(commonPartPercentMinValueForMerge)
        , FeaturesLimit(featuresLimit)
    {}

    void Merge(
            NTemplateMaster::TContextPtr context,
            TStableTemplates<TContentProcessor, TContent> stableTemplates,
            TYield yield) const noexcept
    {
        std::vector<bool> used(stableTemplates.size(), false);
        for (size_t i = 0; i < stableTemplates.size(); i++) {
            for (size_t j = i + 1; j < stableTemplates.size(); j++) {
                if (!used[i] && !used[j]) {
                    const auto templFirst = stableTemplates[i];
                    const auto templSecond = stableTemplates[j];
                    const auto status = TryMergeTemplates(context, templFirst, templSecond, yield);
                    used[i] = status;
                    used[j] = status;
                }
            }
        }
    }

    bool TryMergeTemplates(
            NTemplateMaster::TContextPtr context,
            TStableTemplatePtr<TContentProcessor, TContent> templFirst,
            TStableTemplatePtr<TContentProcessor, TContent> templSecond,
            TYield yield) const noexcept
    {
        using namespace ::NTemplateMaster::NUtils;

        if (templFirst->IsEternal() && templSecond->IsEternal()) {
            return false;
        }
        const auto similarity = CalculateSimilarity(templFirst->GetFeatures(), templSecond->GetFeatures());
        if (similarity >= JacquardMinValueForMerge) {
            auto matchResult = Diff(templFirst, templSecond);
            const auto commonPartPercent = matchResult.GetCommonPartPercent();
            if (commonPartPercent >= CommonPartPercentMinValueForMerge) {
                const auto commonPart = matchResult.GetCommonPart();
                const auto dbTemplate = TUnstableTemplate<TContentProcessor, TContent>(
                        ContentProcessor, commonPart, {}).CreateDatabaseTemplate();
                const auto firstTemplateStableSign = templFirst->GetStableSign();
                const auto secondTemplateStableSign = templSecond->GetStableSign();
                TTemplatesStableSigns stableSigns({firstTemplateStableSign, secondTemplateStableSign});
                LOGDOG_(context->GetLogger(), notice,
                        NTemplateMaster::NLog::type = "TryMergeTemplates",
                        NTemplateMaster::NLog::stable_signs = stableSigns,
                        NTemplateMaster::NLog::stable_sign = dbTemplate->GetStableSign())
                if (dbTemplate->GetStableSign() == firstTemplateStableSign || templFirst->IsEternal()) {
                    const auto res = Db->UpdateSearchFlag(context, false, {secondTemplateStableSign}, yield);
                    if (res) {
                        LOGDOG_(context->GetLogger(), notice,
                                NTemplateMaster::NLog::type = "UpdateSearchFlag",
                                NTemplateMaster::NLog::stable_signs = stableSigns,
                                NTemplateMaster::NLog::stable_sign = secondTemplateStableSign)
                    }
                } else if (dbTemplate->GetStableSign() == secondTemplateStableSign || templSecond->IsEternal()) {
                    const auto res = Db->UpdateSearchFlag(context, false, {firstTemplateStableSign}, yield);
                    if (res) {
                        LOGDOG_(context->GetLogger(), notice,
                                NTemplateMaster::NLog::type = "UpdateSearchFlag",
                                NTemplateMaster::NLog::stable_signs = stableSigns,
                                NTemplateMaster::NLog::stable_sign = firstTemplateStableSign)
                    }
                } else {
                    auto callback = [=](TExpected<bool> result) {
                        if (result && result.value()) {
                            const std::vector<NTemplateMaster::TTemplateStableSign> stableSigns{
                                    firstTemplateStableSign, secondTemplateStableSign};
                            LOGDOG_(context->GetLogger(), notice,
                                    NTemplateMaster::NLog::type = "MergeTemplates",
                                    NTemplateMaster::NLog::stable_signs = stableSigns,
                                    NTemplateMaster::NLog::similarity = similarity,
                                    NTemplateMaster::NLog::common_part_percent = commonPartPercent,
                                    NTemplateMaster::NLog::stable_sign = dbTemplate->GetStableSign())
                        }
                    };
                    Db->AsyncMergeTemplates(context, dbTemplate, FeaturesLimit, TTemplatesStableSigns(stableSigns), callback);
                }
                return true;
            }
        }
        return false;
    }

private:
    NTemplateMaster::NDatabase::TDatabasePtr Db;
    TContentProcessor ContentProcessor;
    const double JacquardMinValueForMerge;
    const double CommonPartPercentMinValueForMerge;
    const int32_t FeaturesLimit;
};

template<typename TContentProcessor, typename TContent>
using TMergeSimilarTemplatesPtr = std::shared_ptr<TMergeSimilarTemplates<TContentProcessor, TContent>>;

}
