#pragma once
#include <mail/template_master/lib/template_master/template_master.h>
#include <mail/template_master/lib/template_master/config.h>
#include <mail/template_master/lib/router/router_module.h>
#include <mail/template_master/lib/db/database_module.h>
#include <mail/template_master/lib/template_master/operations/merge_templates.h>
#include <mail/template_master/lib/template_master/operations/filter_unused_features.h>
#include <mail/template_master/lib/types/match/match_result.h>
#include <mail/template_master/lib/types/operation_result/find_result.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/operation_result/detemple_result.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/template_pool/template_pool.h>

#include <mail/template_master/lib/http/bind_http_handlers.h>

#include <mail/template_master/lib/template_pool/template_pool.h>
#include <mail/template_master/lib/utils/utils.h>

#include <mail/yplatform/include/yplatform/find.h>

#include <sstream>
#include <stdlib.h>
#include <memory>
#include <unordered_map>

namespace NTemplateMaster {

template<typename TContentProcessor, typename TContent, typename TDetempleObjectPtr>
class TTemplateMaster : public ITemplateMaster<TContentProcessor, TContent, TDetempleObjectPtr> {
private:
    using TMergeSimilarTemplatesPtr = TMergeSimilarTemplatesPtr<TContentProcessor, TContent>;
    using TTemplatePtr = TUnstableTemplatePtr<TContentProcessor, TContent>;
    using TCachePtr = NTemplatePool::TTemplateCachePtr<TTemplatePtr>;
    using TTemplatePoolPtr = NTemplatePool::TTemplatePoolPtr
            <TContentProcessor, TContent, TDetempleObjectPtr>;
public:
    TTemplateMaster(TReactor& reactor)
        : Reactor(reactor)
    {}

    void init(yplatform::ptree config) {
        auto templatePoolConfig = yamail::data::deserialization::fromPtree<NTemplatePool::TConfig>
                (config.get_child("template_pool"));
        MaxHashesCount = templatePoolConfig.MaxHashesCount;
        Config = yamail::data::deserialization::fromPtree<TConfig>(config);
        FeaturesLimit = templatePoolConfig.MaxHashesCount;
        ContentProcessor = std::make_shared<TContentProcessor>(Config.MaxEqualTokensInMessage);
        auto db = yplatform::find<NDatabase::IDatabase, std::shared_ptr>("database");
        MergeSimilarTemplates = std::make_shared<TMergeSimilarTemplates<TContentProcessor, TContent>>
                    (db, *ContentProcessor, Config.JacquardMinValueForMerge,
                    Config.CommonPartPercentMinValueForMerge, templatePoolConfig.MaxHashesCount);
        FilterFeatures = std::make_shared<TFilterFeatures>(Reactor.io(), Config.FeaturesUpdateInterval, FeaturesLimit);
        NHttp::BindHttpHandlers(ContentProcessor);
        InitializeTemplatePools(templatePoolConfig);
    }

    // Top N via simlar
    TExpected<TDatabaseTemplates> FindSimilarTemplates(
            NTemplateMaster::TContextPtr context,
            TDetempleObjectPtr message,
            TYield yield) override {
        auto featuresResult = FilterFeatures->Filter(context, message->GetFeatures());
        auto db = yplatform::find<NDatabase::IDatabase>("database");
        if (featuresResult) {
            auto features = featuresResult.value();
            return db->FindSimilarTemplates(context, features, Config.TemplatesLimitFindInDb, FeaturesLimit, yield);
        }
        LOGDOG_(context->GetLogger(), error, NTemplateMaster::NLog::error_code=featuresResult.error())
        return db->FindSimilarTemplates(context, message->GetFeatures(), Config.TemplatesLimitFindInDb, FeaturesLimit, yield);
    }

    void MergeTemplates(
            NTemplateMaster::TContextPtr context,
            TStableTemplates<TContentProcessor, TContent> stableTemplates) override
    {
        auto coro = NTemplateMaster::NUnistat::Wrap([=](TYield yield) mutable {
            MergeSimilarTemplates->Merge(context, stableTemplates, yield);
        }, "application_MergeTemplates");
        boost::asio::spawn(*Reactor.io(), std::move(coro), NTemplateMaster::NUtils::kCoroutineAttributes);
    }

    TContentProcessorPtr<TContentProcessor> GetContentProcessor() const override {
        return ContentProcessor;
    }

    NTemplatePool::TTemplatePoolPtr<TContentProcessor, TContent, TDetempleObjectPtr> FindTemplatePool() override {
        return TemplatePoolWorkers[RandomNumber<uint8_t>() % TemplatePoolWorkers.size()];
    }

    void start() {
        FilterFeatures->Start();
    }

    void stop() {
        auto webServer = yplatform::find<ymod_webserver::server>("web_server");
        webServer->stop();
        FilterFeatures->Stop();
    }

    int GetMaxHashesCount() const override {
        return MaxHashesCount;
    }

private:
    void InitializeTemplatePools(NTemplatePool::TConfig config) {
        auto reactor = yplatform::find_reactor(config.Reactor);
        const auto xorsSize = config.BucketsCount * config.ElementsInBucket;
        TTemplateFeaturesSet xors;
        while (xors.size() < xorsSize) {
            xors.emplace(RandomNumber<uint32_t>());
        }
        size_t number = 0;
        for (auto&& ioPool : reactor->get_pools()) {
            TemplatePoolWorkers.emplace_back(NTemplatePool::MakeTemplatePool
                    <TContentProcessor, TContent, TTemplatePtr, TCachePtr>
                    (ioPool->io(), config, {xors.begin(), xors.end()}, number++));
        }
    }
private:
    TReactor& Reactor;
    TConfig Config;
    size_t FeaturesLimit;
    int MaxHashesCount;
    TContentProcessorPtr<TContentProcessor> ContentProcessor;
    std::vector<TTemplatePoolPtr> TemplatePoolWorkers;
    TMergeSimilarTemplatesPtr MergeSimilarTemplates;
    TFilterFeaturesPtr FilterFeatures;
};
}
