#pragma once
#include <mail/template_master/lib/types/template/message.h>
#include <mail/template_master/lib/template_pool/cache/hash_range.h>

#include <vector>
#include <unordered_map>
#include <unordered_set>
#include <cassert>
#include <limits>

namespace NTemplateMaster::NTemplatePool {

/*
 * https://medium.com/engineering-brainly/locality-sensitive-hashing-explained-304eb39291e4
 */
template<typename TTemplatePtr, typename THasher = THashRange>
class TMinHash {
public:
    using THashType = THasherReturnType<THasher, NTemplateMaster::TTemplateFeaturesSet::const_iterator>;
    using TBuckets = std::vector<std::unordered_map<THashType, std::unordered_set<TTemplateUniqueId>>>;

    TMinHash(
            size_t elementsInBucket,
            size_t bucketsCount,
            NTemplateMaster::TTemplateFeaturesVector xors,
            THasher hasher = THashRange()
    )
        : Xors(std::move(xors))
        , ElementsInBucket(elementsInBucket)
        , BucketsCount(bucketsCount)
        , Hasher(hasher)
    {
        Buckets.resize(BucketsCount, {});
        assert(ElementsInBucket * BucketsCount == Xors.size());
    }

    void InsertTemplate(const TTemplatePtr templ) noexcept {
        const auto minhash = CalculateMinHash(templ);
        const auto templateId = templ->GetUniqueId();
        for (size_t i = 0; i < Buckets.size(); i++) {
            Buckets[i][minhash[i]].insert(templateId);
        }
    }

    template<typename TObjectWithDigestPtr>
    auto FindSimilarTemplates(const TObjectWithDigestPtr obj) const noexcept {
        std::unordered_set<TTemplateUniqueId> result;
        const auto minhash = CalculateMinHash(obj);
        for (size_t i = 0; i < Buckets.size(); i++) {
            const auto& bucket = Buckets[i];
            if (bucket.count(minhash[i])) {
                result.insert(bucket.at(minhash[i]).begin(), bucket.at(minhash[i]).end());
            }
        }
        return result;
    }

    void EraseTemplate(const TTemplatePtr templ) noexcept {
        const auto minhash = CalculateMinHash(templ);
        const auto templateId = templ->GetUniqueId();
        for (size_t i = 0; i < Buckets.size(); i++) {
            Buckets[i][minhash[i]].erase(templateId);
            if (Buckets[i][minhash[i]].empty()) {
                Buckets[i].erase(minhash[i]);
            }
        }
    }

    const auto& GetBuckets() const noexcept {
        return Buckets;
    }

    template<typename TObjectWithDigestPtr>
    auto CalculateMinHash(const TObjectWithDigestPtr object) const noexcept {
        const auto digest = object->GetFeatures();
        std::vector<THashType> minhash;
        minhash.reserve(BucketsCount);
        TTemplateFeaturesVector currentBucketElements;

        for (auto&& xorShinger : Xors) {
            NTemplateMaster::TTemplateFeature minimalHash = std::numeric_limits<NTemplateMaster::TTemplateFeature>::max();
            for (auto&& digestShinger : digest) {
                minimalHash = std::min(minimalHash, digestShinger ^ xorShinger);
            }
            if (!currentBucketElements.empty() && currentBucketElements.size() % ElementsInBucket == 0) {
                minhash.emplace_back(Hasher(currentBucketElements.begin(), currentBucketElements.end()));
                currentBucketElements.clear();
            }
            currentBucketElements.emplace_back(minimalHash);
        }
        minhash.emplace_back(Hasher(currentBucketElements.begin(), currentBucketElements.end()));
        return minhash;
    }
private:
    TBuckets Buckets;
    const NTemplateMaster::TTemplateFeaturesVector Xors;
    const size_t ElementsInBucket;
    const size_t BucketsCount;
    const THasher Hasher;
};

template<typename TTemplatePtr, typename THasher = THashRange>
using TMinHashPtr = std::shared_ptr<TMinHash<TTemplatePtr, THasher>>;

}
