#pragma once
#include <mail/template_master/lib/types/template/message.h>
#include <mail/template_master/lib/template_pool/cache/min_hash.h>

#include <library/cpp/cache/cache.h>

namespace NTemplateMaster::NTemplatePool {

template<typename TTemplatePtr, typename TMinHashPtr = TMinHashPtr<TTemplatePtr>>
class TTemplateCache {
public:
    TTemplateCache(size_t maxCacheSize, TMinHashPtr minHash, size_t similarTemplatesLimit)
        : MaxCacheSize(maxCacheSize)
        , Cache(MaxCacheSize)
        , MinHash(std::move(minHash))
        , SimilarTemplatesLimit(similarTemplatesLimit)
    {}

    auto Size() const noexcept {
        return Cache.Size();
    }

    void InsertTemplate(TTemplatePtr templ) noexcept {
        while (Cache.Size() >= MaxCacheSize) {
            EraseLeastFrequentlyUsedTemplate();
        }
        Cache.Insert(templ->GetUniqueId(), templ);
        MinHash->InsertTemplate(templ);
    }

    auto FindTemplateById(TTemplateUniqueId id) const noexcept -> TOptional<TTemplatePtr> {
        const auto it = Cache.FindWithoutPromote(id);
        if (it == Cache.End()) {
            return {};
        }
        return it.Value();
    }

    template<typename TTokensSequenceType>
    void UpdateTemplate(TTemplatePtr templ, TTokensSequenceType&& newTokens) noexcept {
        MinHash->EraseTemplate(templ);
        templ->UpdateTokens(std::forward<TTokensSequenceType>(newTokens));
        MinHash->InsertTemplate(templ);
    }

    template<typename TObjectWithDigestPtr>
    auto FindSimilarTemplates(const TObjectWithDigestPtr obj) const noexcept {
        auto result = MinHash->FindSimilarTemplates(obj);
        if (result.size() > SimilarTemplatesLimit) {
            auto vec = std::vector(result.begin(), result.end());
            std::random_shuffle(vec.begin(), vec.end());
            vec.resize(SimilarTemplatesLimit);
            result = {vec.begin(), vec.end()};
        }
        return result;
    }

    void PromoteTemplate(TTemplatePtr templ) noexcept {
        Cache.Find(templ->GetUniqueId());
    }

    void EraseTemplate(TTemplatePtr templ) noexcept {
        MinHash->EraseTemplate(templ);
        Cache.Erase(Cache.FindWithoutPromote(templ->GetUniqueId()));
    }
private:
    void EraseLeastFrequentlyUsedTemplate() noexcept {
        const auto leastFreqUsed = Cache.FindLeastFrequentlyUsed();
        if (leastFreqUsed != Cache.End()) {
            EraseTemplate(leastFreqUsed.Value());
        }
    }

private:
    const size_t MaxCacheSize;
    TLFUCache<TTemplateUniqueId, TTemplatePtr> Cache;
    TMinHashPtr MinHash;
    const size_t SimilarTemplatesLimit;
};

template<typename TTemplatePtr, typename TMinHashPtr = TMinHashPtr<TTemplatePtr>>
using TTemplateCachePtr = std::shared_ptr<TTemplateCache<TTemplatePtr, TMinHashPtr>>;

template<typename TTemplatePtr>
inline auto MakeTemplateCache(
        size_t maxCacheSize,
        size_t elementsInBucket,
        size_t bucketsCount,
        NTemplateMaster::TTemplateFeaturesVector xors,
        size_t similarTemplatesLimit)
{
    auto minHash = std::make_shared<TMinHash<TTemplatePtr>>(elementsInBucket, bucketsCount, std::move(xors));
    return std::make_shared<TTemplateCache<TTemplatePtr>>(maxCacheSize, minHash, similarTemplatesLimit);
}

}
