#pragma once
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/log.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/types/template/unstable_template.h>
#include <mail/template_master/lib/unistat/unistat.h>

namespace NTemplateMaster::NTemplatePool {

template<typename TContentProcessor, typename TContent>
class TCalculateBestMatchOperation {
private:
    using TTokenType = ContentProcessorTokenType<TContentProcessor, TContent>;
    static constexpr double maxSmoothingCoeff = 100.0;
    static constexpr double maxCommonPartPercent = 0.95;
public:
    template<typename TTemplate>
    using TMatchTemplate = std::pair<
        TTemplate,
        TMatchResult<ContentProcessorTokenType<TContentProcessor, TContent>>
    >;

    TCalculateBestMatchOperation(
            double commonPartSmoothingCoefficient,
            double commonPartPercentForMatch,
            size_t poolId
    )
        : CommonPartSmoothingCoefficient(commonPartSmoothingCoefficient)
        , CommonPartPercentForMatch(commonPartPercentForMatch)
        , PoolId(poolId)
    {}

    template<typename TTemplate>
    auto GetBestMatch(
            TContextPtr context,
            const TMessagePtr<TContentProcessor, TContent> msg,
            const std::vector<TTemplate>& templates)
            const noexcept -> TOptional<TMatchTemplate<TTemplate>>
    {
        for (auto&& templ : templates) {
            const auto match = NTemplateMaster::NUnistat::WrapWithLog([msg, templ]() {
                return NUtils::Diff(msg, templ);
            }, context, "application_pool_Diff")();
            if (CheckIsTemplateMatches(context, templ, match)) {
                LOGDOG_(context->GetDebugLogger(), debug,
                        NTemplateMaster::NLog::type="MatchTemplate",
                        NTemplateMaster::NLog::body=templ->Json())
                return std::make_pair(templ, match);
            }
        }
        return {};
    }

    template<typename TTemplate>
    auto CheckIsTemplateMatches(
            TContextPtr context,
            const TTemplate& templ,
            const TMatchResult<TTokenType>& match) const noexcept -> bool
    {
        const auto hits = templ->GetHits();
        const double smoothingCoeff = std::min(maxSmoothingCoeff, pow(CommonPartSmoothingCoefficient, hits));
        const double threshold = std::min(maxCommonPartPercent, CommonPartPercentForMatch * smoothingCoeff);
        const double commonPartPercent = match.GetCommonPartPercent();
        LOGDOG_(context->GetLogger(), notice,
                NTemplateMaster::NLog::type="CheckIsTemplateMatches",
                NTemplateMaster::NLog::hits=hits,
                NTemplateMaster::NLog::smoothing_coeff=smoothingCoeff,
                NTemplateMaster::NLog::threshold=threshold,
                NTemplateMaster::NLog::pool_id=PoolId,
                NTemplateMaster::NLog::common_part_percent=commonPartPercent)
        return commonPartPercent >= threshold;
    }
private:
    const double CommonPartSmoothingCoefficient;
    const double CommonPartPercentForMatch;
    const size_t PoolId;
};

template<typename TContentProcessor, typename TContent>
using TCalculateBestMatchOperationPtr = std::shared_ptr<TCalculateBestMatchOperation
        <TContentProcessor, TContent>>;

}
