#pragma once
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/log.h>
#include <mail/template_master/lib/types/template/unstable_template.h>
#include <mail/template_master/lib/unistat/unistat.h>
#include <mail/template_master/lib/template_pool/cache/template_cache.h>
#include <mail/template_master/lib/utils/utils.h>

namespace NTemplateMaster::NTemplatePool {

template<typename TTemplateCachePtr, typename TTemplatePtr>
class TGetCandidateTemplatesOperation {
public:
    TGetCandidateTemplatesOperation(
            TTemplateCachePtr templateCache,
            size_t maxTemplatesAfterFilter,
            double jacquardMinValueForDetemple,
            size_t poolId
    )
        : TemplateCache(std::move(templateCache))
        , MaxTemplatesAfterFilter(maxTemplatesAfterFilter)
        , JacquardMinValueForDetemple(jacquardMinValueForDetemple)
        , PoolId(poolId)
    {}

    template<typename TObjectWithDigestPtr>
    auto GetCandidateTemplates(TContextPtr context, TObjectWithDigestPtr msg) const noexcept {
        const auto similarTemplates = NTemplateMaster::NUnistat::Wrap([=]() {
            return TemplateCache->FindSimilarTemplates(msg);
        }, "application_pool_FindSimilarTemplates")();
        auto unistat = yplatform::find<NYmodUnistat::IUnistat>("unistat");
        unistat->Push("application_pool_similar_templates_count", similarTemplates.size());
        const auto candidate_templates = FilterTemplates(context, similarTemplates, msg);
        unistat->Push("application_pool_filter_similar_templates_count", candidate_templates.size());
        LogCandidateTemplates(context, candidate_templates);
        return candidate_templates;
    }

    template<typename TObjectWithDigestPtr>
    auto FilterTemplates(
            TContextPtr context,
            const std::unordered_set<TTemplateUniqueId>& templatesIds,
            TObjectWithDigestPtr msg) const noexcept
    {
        std::vector<std::pair<TSimilarity, TTemplatePtr>> result;
        const auto& digest = msg->GetFeatures();

        for (auto&& id : templatesIds) {
            const auto templateOpt = TemplateCache->FindTemplateById(id);
            if (templateOpt) {
                const auto templ = templateOpt.value();
                const auto& template_digest  = templ->GetFeatures();
                const auto similarity = NUtils::CalculateSimilarity(digest, template_digest);
                LOGDOG_(context->GetLogger(), notice,
                        NTemplateMaster::NLog::type="Similarity",
                        NTemplateMaster::NLog::similarity=similarity,
                        NTemplateMaster::NLog::pool_id=PoolId)
                if (similarity > JacquardMinValueForDetemple) {
                    result.emplace_back(similarity, templ);
                }
            } else {
                const std::string message = "Template " + std::to_string(id) + " not found in cache";
                LOGDOG_(context->GetLogger(), error, NTemplateMaster::NLog::message=message)
            }
        }
        std::sort(result.begin(), result.end(), [](auto&& v1, auto&& v2) {
            return v1.first > v2.first;
        });
        std::vector<TTemplatePtr> templates;
        std::transform(result.begin(), result.end(), std::back_inserter(templates), [](auto&& val) {
            return val.second;
        });
        if (templates.size() > MaxTemplatesAfterFilter) {
            templates.resize(MaxTemplatesAfterFilter);
        }
        return templates;
    }

    void LogCandidateTemplates(TContextPtr context, std::vector<TTemplatePtr> templates) const noexcept {
        std::vector<NTemplateMaster::TTemplateUniqueId> templateUniqueIds;
        std::transform(templates.begin(), templates.end(),
                std::back_inserter(templateUniqueIds), [](auto&& t) {
                    return t->GetUniqueId();
        });
        LOGDOG_(context->GetLogger(), notice,
                NTemplateMaster::NLog::type="CandidateTemplates",
                NTemplateMaster::NLog::templates_unique_ids=templateUniqueIds,
                NTemplateMaster::NLog::pool_id=PoolId)
    }
private:
    TTemplateCachePtr TemplateCache;
    const size_t MaxTemplatesAfterFilter;
    const double JacquardMinValueForDetemple;
    const size_t PoolId;
};

template<typename TTemplateCachePtr, typename TTemplatePtr>
using TGetCandidateTemplatesOperationPtr = std::shared_ptr<
        TGetCandidateTemplatesOperation<TTemplateCachePtr, TTemplatePtr>>;

}
