#pragma once
#include <mail/template_master/lib/template_pool/cache/template_cache.h>
#include <mail/template_master/lib/template_pool/config.h>
#include <mail/template_master/lib/template_pool/operations/save_ready_template.h>
#include <mail/template_master/lib/template_pool/operations/calculate_best_match.h>
#include <mail/template_master/lib/template_pool/operations/get_candidate_templates.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/types/operation_result/detemple_result.h>
#include <mail/template_master/lib/types/template/message.h>
#include <mail/template_master/lib/types/token/token.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/unistat/unistat.h>

#include <boost/range/algorithm/transform.hpp>

#include <mail/logdog/include/logdog/logger.h>

namespace NTemplateMaster::NTemplatePool {

template<typename TContentProcessor, typename TContent, typename TDetempleObjectPtr, typename TDbHints>
class ITemplatePool {
public:
    using TAttributes = ContentProcessorAttributesType<TContentProcessor, TContent>;
    using TTokenType = ContentProcessorTokenType<TContentProcessor, TContent>;

    virtual ~ITemplatePool() = default;

    virtual TDetempleResult<TTokenType, TAttributes> AddToPool(
            const TContextPtr context,
            const TDetempleObjectPtr msg,
            std::optional<TDbHints> dbHints,
            TYield yld) = 0;
};

template<typename TContentProcessor,
        typename TContent,
        typename TTemplatePtr,
        typename TCachePtr,
        typename TDetempleObjectPtr = TMessagePtr<TContentProcessor, TContent>,
        typename TCalculateBestMatchOperationPtr = TCalculateBestMatchOperationPtr<TContentProcessor, TContent>,
        typename TSaveReadyTemplateOperationPtr = TSaveReadyTemplateOperationPtr<TContentProcessor, TContent>,
        typename TGetCandidateTemplatesOperationPtr = TGetCandidateTemplatesOperationPtr<TCachePtr, TTemplatePtr>
>
class TTemplatePool : public ITemplatePool<TContentProcessor, TContent, TDetempleObjectPtr, TDbHints>,
        public std::enable_shared_from_this<TTemplatePool<TContentProcessor,
        TContent, TTemplatePtr, TCachePtr, TDetempleObjectPtr, TCalculateBestMatchOperationPtr,
        TSaveReadyTemplateOperationPtr, TGetCandidateTemplatesOperationPtr>> {
public:
    static_assert(ContentProcessorConcept<TContentProcessor, TContent>, "must satisfy ContentProcessor concept");
    using typename ITemplatePool<TContentProcessor, TContent, TDetempleObjectPtr, TDbHints>::TAttributes;
    using typename ITemplatePool<TContentProcessor, TContent, TDetempleObjectPtr, TDbHints>::TTokenType;

    TTemplatePool(
            boost::asio::io_service* io,
            TConfig config,
            size_t poolId,
            TCalculateBestMatchOperationPtr calculateBestMatchOp,
            TSaveReadyTemplateOperationPtr saveReadyTemplateOp,
            TCachePtr cache,
            TGetCandidateTemplatesOperationPtr getCandidateTemplatesOp
    )
        : Strand(*io)
        , Conf(std::move(config))
        , SaveOperation(std::move(saveReadyTemplateOp))
        , CalculateBestMatchOp(std::move(calculateBestMatchOp))
        , TemplateCache(std::move(cache))
        , GetCandidateTemplatesOp(std::move(getCandidateTemplatesOp))
        , PoolId(poolId)
    {}

    TDetempleResult<TTokenType, TAttributes> AddToPool(
            const TContextPtr context,
            const TDetempleObjectPtr msg,
            std::optional<TDbHints> dbHints,
            TYield yld) override
    {
        TAsyncCompletion<void(TDetempleResult<TTokenType, TAttributes>)> init(yld);
        const auto handler = init.completion_handler;
        auto coro = [self=SharedFromThis(), context, msg, handler, dbHints=std::move(dbHints)](TYield) mutable {
            auto result = self->Detemple(context, msg, std::move(dbHints));
            NUtils::Dispatch(handler, result);
        };
        boost::asio::spawn(Strand, std::move(coro), NUtils::kCoroutineAttributes);
        return init.result.get();
    }
private:
    decltype(auto) SharedFromThis() noexcept {
        return this->shared_from_this();
    }

    static auto AdaptDatabaseTemplates(TDbHints dbHints, TContentProcessor contentProcessor, size_t matchesForReadyTemplate) {
        using TAdaptor = TDatabaseTemplateDetempleAdaptor<TContentProcessor, TContent>;
        std::vector<std::shared_ptr<TAdaptor>> result;
        result.reserve(dbHints.size());
        std::transform(
                std::make_move_iterator(dbHints.begin()),
                std::make_move_iterator(dbHints.end()),
                std::back_inserter(result),
                [&contentProcessor, &matchesForReadyTemplate](auto&& dbHint) {
                    return std::make_shared<TAdaptor>(
                            MakeDatabaseTemplate(std::move(dbHint)),
                            contentProcessor,
                            matchesForReadyTemplate
                            );
                });
        return result;

    }

    template <typename TCandidateTemplates>
    auto FindBestMatch(const TCandidateTemplates& candidates, TContextPtr context, TDetempleObjectPtr msg) {
        auto bestMatch = NTemplateMaster::NUnistat::Wrap(
        [this, &context, &msg, &candidates]() {
            return CalculateBestMatchOp->GetBestMatch(context, msg, candidates);
        }, "application_pool_CalculateBestMatch")();
        return bestMatch;
    }

    template <typename TTemplate>
    TDetempleResult<TTokenType, TAttributes> MakeResponse(
            EDetempleStatus detempleStatus,
            TMatchResult<TTokenType> matchResult,
            const TTemplate& matchingTemplate)
    {
        switch (detempleStatus) {
            case EDetempleStatus::FoundInDbHints:
            case EDetempleStatus::FoundPreparedInPool:
                return {detempleStatus, matchResult.GetDelta(), matchingTemplate->GetAttributesArray(),
                        matchingTemplate->GetStableSign()};
            case EDetempleStatus::FoundNotReadyInPool:
                return {detempleStatus, matchResult.GetDelta(), matchingTemplate->GetAttributesArray()};
            case EDetempleStatus::NotFound:
                return {detempleStatus};
            default:
                Y_FAIL();
        }
    }

    TDetempleResult<TTokenType, TAttributes> Detemple(
            const TContextPtr context,
            const TDetempleObjectPtr msg,
            std::optional<TDbHints> dbHints) noexcept {
        auto candidatesFromPool = NTemplateMaster::NUnistat::Wrap(
        [self=SharedFromThis(), context, msg]() {
            return self->GetCandidateTemplatesOp->GetCandidateTemplates(context, msg);
        }, "application_pool_GetCandidateTemplates")();

        auto bestMatchFromPool = FindBestMatch(candidatesFromPool, context, msg);
        if (bestMatchFromPool.has_value()) {
            auto&& [bestMatchTemplate, bestMatchResult] = bestMatchFromPool.value();
            if (bestMatchTemplate->GetHits() >= Conf.MatchesForReadyTemplate) {
                auto db = yplatform::find<NTemplateMaster::NDatabase::IDatabase, std::shared_ptr>("database");
                SaveOperation->Save(context, bestMatchTemplate, db);
                return MakeResponse(EDetempleStatus::FoundPreparedInPool, bestMatchResult, bestMatchTemplate);
            } else {
                UpdateMatchingTemplate(bestMatchTemplate, bestMatchResult, msg);
                return MakeResponse(EDetempleStatus::FoundNotReadyInPool, bestMatchResult, bestMatchTemplate);
            }
        } else if (dbHints.has_value()) {
            LOGDOG_(context->GetLogger(), notice, NLog::message="no matches found in pool; let's look up in the db hints")
            TContentProcessor contentProcessor = msg->GetContentProcessor();
            auto adaptedDbTemplates = AdaptDatabaseTemplates(std::move(dbHints).value(), contentProcessor, Conf.MatchesForReadyTemplate);
            auto bestMatchFromDb = FindBestMatch(adaptedDbTemplates, context, msg);
            if (bestMatchFromDb.has_value()) {
                auto&& [bestMatchTemplate, bestMatchResult] = bestMatchFromDb.value();
                return MakeResponse(EDetempleStatus::FoundInDbHints, bestMatchResult, bestMatchTemplate);
            }
        }

        auto templ = msg->CreateUnstableTemplate();
        NTemplateMaster::NUnistat::Wrap(
        [self=SharedFromThis(), templ=std::move(templ)]() {
            self->TemplateCache->InsertTemplate(templ);
        }, "application_pool_InsertTemplate")();
        return {EDetempleStatus::NotFound};
    }

    void UpdateMatchingTemplate(
            const TTemplatePtr bestMatchTemplate,
            const TMatchResult<TTokenType>& bestMatch,
            const TDetempleObjectPtr msg) noexcept
    {
        TemplateCache->PromoteTemplate(bestMatchTemplate);
        const auto newTokens = bestMatch.GetCommonPart();
        TemplateCache->UpdateTemplate(bestMatchTemplate, newTokens);
        bestMatchTemplate->PushAttributes(msg->GetAttributes());
        bestMatchTemplate->IncrementHits();
    }

    size_t GetId() const noexcept {
        return PoolId;
    }
private:
    TStrand Strand;
    const TConfig Conf;
    const TSaveReadyTemplateOperationPtr SaveOperation;
    const TCalculateBestMatchOperationPtr CalculateBestMatchOp;
    const TCachePtr TemplateCache;
    const TGetCandidateTemplatesOperationPtr GetCandidateTemplatesOp;
    const size_t PoolId;
};

template<typename TContentProcessor, typename TContent, typename TDetempleObjectPtr>
using TTemplatePoolPtr = std::shared_ptr<ITemplatePool<TContentProcessor, TContent, TDetempleObjectPtr, TDbHints>>;

template<typename TContentProcessor, typename TContent, typename TTemplatePtr, typename TCachePtr>
inline auto MakeTemplatePool(
        boost::asio::io_service* io,
        TConfig config,
        TTemplateFeaturesVector xors,
        size_t poolId)
{
    const auto сalculateBestMatchOp = std::make_shared<TCalculateBestMatchOperation<TContentProcessor, TContent>>(
            config.CommonPartSmoothingCoefficient, config.CommonPartPercentForMatch, poolId);
    const auto saveReadyTemplateOp = std::make_shared<TSaveReadyTemplateOperation<TContentProcessor, TContent>>(config.MaxHashesCount);
    const auto cache = MakeTemplateCache<TTemplatePtr>(
            config.MaxCacheSize, config.ElementsInBucket, config.BucketsCount, xors, config.SimilarTemplatesLimit);
    const auto getCandidateTemplatesOp = std::make_shared<TGetCandidateTemplatesOperation<TCachePtr, TTemplatePtr>>(
            cache, config.MaxTemplatesAfterFilter, config.JacquardMinValueForDetemple, poolId);
    return std::make_shared<NTemplatePool::TTemplatePool<TContentProcessor, TContent, TTemplatePtr, TCachePtr>>(
            io, config, poolId, сalculateBestMatchOp, saveReadyTemplateOp, cache, getCandidateTemplatesOp);
}

}
