#pragma once
#include <mail/template_master/lib/types/token/token.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/types/template/unstable_template.h>
#include <mail/template_master/lib/types/match/delta.h>

namespace NTemplateMaster {

template<typename TTokenType>
class TMatchResult {
public:
    TMatchResult(TChunks<TTokenType> diff)
        : Diff(std::move(diff))
        , FullMatch(CheckFullMatch())
        , CommonPart(ComputeCommonPart())
        , CommonPartPercent(ComputePercentOfCommonPartInTemplate())
        , Delta(ComputeDelta())
    {}

    auto IsFullMatch() const noexcept {
        return FullMatch;
    }

    const auto& GetDelta() const noexcept {
        return Delta;
    }

    const auto& GetDiff() const & noexcept {
        return Diff;
    }

    auto GetCommonPartPercent() const noexcept {
        return CommonPartPercent;
    }

    const auto& GetCommonPart() const & noexcept {
        return CommonPart;
    }

private:
    bool CheckFullMatch() const noexcept {
        for (auto&& chunk : Diff) {
            if (!chunk.Left.empty() || !chunk.Right.empty()) {
                if (!(chunk.Right.size() == 1 && chunk.Right.front().IsSentinel())) {
                    return false;
                }
            }
        }
        return true;
    }

    TDelta<TTokenType> ComputeDelta() const noexcept {
        TDelta<TTokenType> delta;
        for (auto&& chunk : Diff) {
            if (!chunk.Left.empty() || !chunk.Right.empty()) {
                delta.emplace_back();
                for (auto&& token : chunk.Left) {
                    delta.back().emplace_back(token.GetValue());
                }
            }
        }
        return delta;
    }

    TTokenSequence<TTokenType> ComputeCommonPart() const noexcept {
        TTokenSequence<TTokenType> result;
        result.emplace_back();
        for (auto&& chunk : Diff) {
            if ((!chunk.Left.empty() || !chunk.Right.empty()) && !result.back().IsSentinel()) {
                result.emplace_back();
            }
            if (!chunk.Common.empty()) {
                for (auto&& token : chunk.Common) {
                    result.emplace_back(token);
                }
            }
        }
        if (!result.back().IsSentinel()) {
            result.emplace_back();
        }
        return result;
    }

    double ComputePercentOfCommonPartInTemplate() const noexcept {
        size_t commonTokensCount = 0, templateTokensCount = 0;
        for (auto&& chunk : Diff) {
            commonTokensCount += chunk.Common.size();
            templateTokensCount += chunk.Common.size();
            for (auto&& templ_token : chunk.Right) {
                if (!templ_token.IsSentinel()) {
                    templateTokensCount++;
                }
            }
        }
        return templateTokensCount == 0 ? 0 : static_cast<double>(commonTokensCount) / templateTokensCount;
    }

    TChunks<TTokenType> Diff;
    bool FullMatch;
    TTokenSequence<TTokenType> CommonPart;
    double CommonPartPercent;
    TDelta<TTokenType> Delta;
};

template<typename TContentProcessor, typename TContent>
using TMatchStableTemplate = std::pair<TStableTemplatePtr<TContentProcessor, TContent>, TMatchResult<ContentProcessorTokenType<TContentProcessor, TContent>>>;

template<typename TContentProcessor, typename TContent>
using TMatchStableTemplates = std::vector<TMatchStableTemplate<TContentProcessor, TContent>>;

template<typename TContentProcessor, typename TContent>
using TMatchUnstableTemplate = std::pair<TUnstableTemplatePtr<TContentProcessor, TContent>, TMatchResult<ContentProcessorTokenType<TContentProcessor, TContent>>>;

template<typename TContentProcessor, typename TContent>
using TMatchUnstableTemplates = std::vector<TMatchUnstableTemplate<TContentProcessor, TContent>>;
}
