#pragma once
#include <mail/template_master/lib/types/operation_request/email_http_request.h>
#include <mail/template_master/lib/types/template/stable_template.h>
#include <mail/template_master/lib/types/template/stable_sign.h>
#include <mail/template_master/lib/types/template/template_sign.h>
#include <mail/template_master/lib/types/template/attributes.h>

#include <mail/yreflection/include/yamail/data/deserialization/yajl.h>

#include <string>

namespace NTemplateMaster {

class TDatabaseTemplate : public TUniqueObject {
public:
    TDatabaseTemplate(
            TTemplateStableSign stableSign,
            TTemplateFeaturesSet features,
            std::string json,
            TJsonAttributesArray attributesArray)
        : StableSign(stableSign)
        , Features(std::move(features))
        , JsonValue(std::move(json))
        , JsonAttributesArray(std::move(attributesArray))
    {}

    const auto& Json() const & noexcept {
        return JsonValue;
    }

    const auto& GetJsonAttributesArray() const & noexcept {
        return JsonAttributesArray;
    }

    const auto& GetFeatures() const & noexcept {
        return Features;
    }

    auto GetStableSign() const noexcept {
        return StableSign;
    }

    template<typename TContentProcessor, typename TContent>
    auto CreateStableTemplate(TContentProcessor tokenizer) const noexcept {
        static_assert(ContentProcessorConcept<TContentProcessor, TContent>, "must satisfy ContentProcessor concept");

        auto attributesStr = JsonAttributesArray.raw_string();
        TAttributesArray<ContentProcessorAttributesType<TContentProcessor, TContent>> attributesArray;

        bool isEternal = false;
        if (!attributesStr.empty()) {
            yamail::data::deserialization::fromJson(attributesStr, attributesArray);
            for (auto& attributes : attributesArray) {
                if (attributes.IsEternal()) {
                    isEternal = true;
                    break;
                }
            }
        }

        return std::make_shared<TStableTemplate<TContentProcessor, TContent>>(std::move(tokenizer), JsonValue, isEternal);
    }
private:
    const TTemplateStableSign StableSign;
    const TTemplateFeaturesSet Features;
    const std::string JsonValue;
    const TJsonAttributesArray JsonAttributesArray;
};

using TDatabaseTemplatePtr = std::shared_ptr<TDatabaseTemplate>;
using TDatabaseTemplates = std::vector<TDatabaseTemplatePtr>;

template <typename TContentProcessor, typename TContent>
class TDatabaseTemplateDetempleAdaptor {
public:
    TDatabaseTemplateDetempleAdaptor(TDatabaseTemplate dbTemplate, TContentProcessor contentProcessor, size_t matchesForReadyTemplate)
            : DbTemplate(std::move(dbTemplate)),
              StableTemplate(DbTemplate.template CreateStableTemplate<TContentProcessor, TContent>(contentProcessor)),
              MatchesForReadyTemplate(matchesForReadyTemplate)
    {}

    auto GetAttributesArray() {
        auto attributesStr = DbTemplate.GetJsonAttributesArray().raw_string();
        TAttributesArray<ContentProcessorAttributesType<TContentProcessor, TContent>> attributes;
        if (!attributesStr.empty()) {
            yamail::data::deserialization::fromJson(attributesStr, attributes);
        }
        return attributes;
    }

    decltype(auto) GetFeatures() const noexcept {
        return DbTemplate.GetFeatures();
    }

    auto GetHits() const noexcept {
        return MatchesForReadyTemplate;
    }

    auto GetStableSign() const noexcept {
        return DbTemplate.GetStableSign();
    }

    decltype(auto) GetTokens() const noexcept {
        return StableTemplate->GetTokens();
    }

    decltype(auto) Json() const noexcept {
        return DbTemplate.Json();
    }

private:
    TDatabaseTemplate DbTemplate;
    TStableTemplatePtr<TContentProcessor, TContent> StableTemplate;
    const size_t MatchesForReadyTemplate;
};

template <typename TContentProcessor, typename TContent>
using TDatabaseTemplateDetempleAdaptorPtr = std::shared_ptr<TDatabaseTemplateDetempleAdaptor<TContentProcessor, TContent>>;

inline TDatabaseTemplate MakeDatabaseTemplate(TDbHint dbHint) {
    auto features = TTemplateFeaturesSet{dbHint.Features.begin(), dbHint.Features.end()};
    return {dbHint.StableSign,
            std::move(features),
            std::move(dbHint.Tokens),
            TJsonAttributesArray{std::move(dbHint.Attributes)}};
}

}
