#pragma once
#include <mail/template_master/lib/types/token/token_traits.h>
#include <mail/template_master/lib/types/optional.h>

#include <string>

namespace NTemplateMaster {

template<typename TValue>
class TToken {
public:
    static_assert(TokenValueConcept<TValue>, "must satisfy TokenValueConcept");
    using TTokenValue = TValue;

    template<typename T>
    friend const T& GetTokenValue(const TToken<T>& token);

    TToken() = default;

    TToken(TValue value)
        : Value(std::move(value))
    {}

    decltype(auto) GetValue() const & noexcept {
        return GetTokenValue(*this);
    }

    bool IsSentinel() const noexcept {
        return !Value.has_value();
    }

    size_t Hash() const noexcept {
        return std::hash<TValue>()(GetValue());
    }

    bool operator==(const TToken& other) const noexcept {
        return IsSentinel() ? false : Value == other.Value;
    }

    bool operator!=(const TToken& other) const noexcept {
        return !(*this == other);
    }

    auto operator+(const TToken& rhs) const {
        if (IsSentinel() || rhs.IsSentinel()) {
            throw std::runtime_error("Can't add sentinel");
        }
        return TToken(Value.value() + rhs.Value.value());
    }

    auto& operator+=(const TToken& rhs) {
        if (IsSentinel() || rhs.IsSentinel()) {
            throw std::runtime_error("Can't add sentinel");
        }
        Value.value() += rhs.GetValue();
        return *this;
    }
private:
    TOptional<TValue> Value;
};

template<typename TTokenType, typename = std::enable_if_t<TokenConcept<TTokenType>, void>>
using TTokenSequence = std::vector<TTokenType>;

template<typename TTokenType, typename = std::enable_if_t<TokenConcept<TTokenType>, void>>
using TTokenView = std::reference_wrapper<const TTokenType>;

template<typename TTokenType, typename = std::enable_if_t<TokenConcept<TTokenType>, void>>
class TTokenSequenceView : public std::vector<TTokenView<TTokenType>> {
public:
    TTokenSequenceView(const TTokenSequence<TTokenType>& seq)
        : std::vector<TTokenView<TTokenType>>(seq.begin(), seq.end())
    {}
};

}
