#pragma once
#include <boost/system/error_code.hpp>

namespace NTemplateMaster::NErrors {

using TErrorCode = ::boost::system::error_code;
using TErrorCategory = ::boost::system::error_category;

enum class EError {
    RouterNotReady = 1,
    RequestedDbNotFound = 2,
    TaskNotFound = 3,
    UnknownTaskType = 4,
    CacheNotInited = 5,
};

class TCategory: public boost::system::error_category {
public:
    const char* name() const noexcept override {
        return "NTemplateMaster::NErrors::TCategory";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::RouterNotReady:
                return "router not ready";
            case EError::RequestedDbNotFound:
                return "database with requested execution policy not exists";
            case EError::TaskNotFound:
                return "Task with requested id not found";
            case EError::UnknownTaskType:
                return "Unknown task type";
            case EError::CacheNotInited:
                return "Cache not inited";
        }

        return "unknown error (" + std::to_string(value) + ")";
    }
};

inline const TErrorCategory& category() noexcept {
    static TCategory instance;
    return instance;
}

inline auto MakeErrorCode(const EError e) {
    return boost::system::error_code(static_cast<int>(e), category());
}

}

namespace boost::system {

template <>
struct is_error_code_enum<NTemplateMaster::NErrors::EError>: public std::true_type {};

}

