#pragma once
#include <mail/template_master/lib/template_master/content_processor/shinger_print.h>
#include <mail/template_master/lib/types/asio.h>
#include <mail/template_master/lib/types/match/match_result.h>
#include <mail/template_master/lib/types/match/similarity.h>
#include <mail/template_master/lib/types/template/message.h>
#include <mail/template_master/lib/types/token/token.h>

#include <boost/algorithm/string/join.hpp>
#include <boost/asio/dispatch.hpp>
#include <boost/coroutine/attributes.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <util/stream/file.h>
#include <util/folder/dirut.h>
#include <util/generic/string.h>

#include <library/cpp/diff/diff.h>

#include <string>
#include <unordered_set>

namespace NTemplateMaster::NUtils {

static constexpr size_t coroutineStackSize = 1e8;
static boost::coroutines::attributes kCoroutineAttributes(coroutineStackSize);

inline auto CalculateSimilarity(
        const TTemplateFeaturesSet& lhs,
        const TTemplateFeaturesSet& rhs) noexcept -> NTemplateMaster::TSimilarity
{
    const auto intersectionSize = std::count_if(lhs.cbegin(), lhs.cend(), [&](auto&& element) {
        return rhs.count(element);
    });
    return intersectionSize ?
        static_cast<NTemplateMaster::TSimilarity>(intersectionSize) /
        (lhs.size() + rhs.size() - intersectionSize) : 0;
}

template<typename TTokenType>
void InlineDiff(
        TChunks<TTokenType>& chunks,
        const TConstArrayRef<TTokenType>& left,
        const TConstArrayRef<TTokenType>& right) noexcept
{
    static_assert(TokenConcept<TTokenType>, "must satisfy TokenConcept");
    TVector<NDiff::TChunk<TTokenType>> resultChunks;
    NDiff::InlineDiff(resultChunks, left, right);
    chunks = std::move(resultChunks);
}

template <typename TLhsTemplatePtr, typename TRhsTemplatePtr>
inline auto Diff(TLhsTemplatePtr lhs, TRhsTemplatePtr rhs) noexcept {
    using TTokenType = std::decay_t<decltype(*std::declval<TLhsTemplatePtr>()->GetTokens().begin())>;
    TChunks<TTokenType> result;
    InlineDiff(
        result,
        TConstArrayRef<TTokenType>(lhs->GetTokens()),
        TConstArrayRef<TTokenType>(rhs->GetTokens()));
    return NTemplateMaster::TMatchResult<TTokenType>(result);
}

template <typename TToken>
auto MatchDbTemplate(
    const TTokenSequence<TToken>& msgTokens,
    const TTokenSequence<TToken>& templTokens) -> TOptional<TDelta<TToken>>
{
    TTokenSequenceView<TToken> allTokens(templTokens);
    allTokens.insert(allTokens.end(), msgTokens.begin(), msgTokens.end());
    size_t link = 0;
    int lastMatchPos = -1;
    bool isFullMatch = false;
    std::vector<size_t> links(allTokens.size(), 0);
    TDelta<TToken> delta;
    for (size_t pos = 0; pos < allTokens.size(); pos++) {
        if (pos == templTokens.size() - 1) {
            link = 0;
            links[pos] = pos;
            continue;
        }
        if (allTokens[pos].get().IsSentinel()) {
            links[pos] = pos;
            links[pos + 1] = pos;
            link = pos;
            pos++;
            continue;
        }
        while (!allTokens[link].get().IsSentinel() &&
                allTokens[link + 1].get() != allTokens[pos].get()) {
            link = links[link];
        }
        if (allTokens[link + 1].get() == allTokens[pos].get()) {
            link++;
        }
        links[pos] = link;
        if (allTokens[link + 1].get().IsSentinel()) {
            size_t startPos = 0, endPos;
            if (lastMatchPos == -1) {
                startPos = templTokens.size();
                endPos = pos - link + 1;
            } else {
                startPos = lastMatchPos + 1;
                endPos = pos - (link - links[lastMatchPos]) + 2;
            }
            delta.emplace_back();
            if (startPos < endPos) {
                std::transform(allTokens.begin() + startPos, allTokens.begin() + endPos,
                    std::back_inserter(delta.back()), [](auto&& tokenView) {
                      return tokenView.get().GetValue();
                    });
            }
            lastMatchPos = pos;
            link++;
        }
        if (link == templTokens.size() - 1) {
            delta.emplace_back();
            if (pos + 1 < allTokens.size()) {
                std::transform(allTokens.begin() + pos + 1, allTokens.end(),
                    std::back_inserter(delta.back()), [](auto&& tokenView) {
                      return tokenView.get().GetValue();
                    });
            }
            isFullMatch = true;
            break;
        }
    }
    if (isFullMatch) {
        return delta;
    }
    return TOptional<TDelta<TToken>>{};
}

template <typename TDetempleObjectPtr, typename TContentProcessor, typename TContent>
auto MatchDbTemplates(
        TDetempleObjectPtr msg,
        TStableTemplates<TContentProcessor, TContent> dbTemplates)
{
    TMatchStableTemplates<TContentProcessor, TContent> matches;
    for (auto&& templ : dbTemplates) {
        matches.emplace_back(templ, NUtils::Diff(msg, templ));
    }
    return matches;
}

template <typename TCompletionToken, typename... TArgs>
void Dispatch(TCompletionToken&& handler, TArgs&&... args) {
    auto executor = boost::asio::get_associated_executor(handler);
    boost::asio::dispatch(executor,
            std::bind(std::forward<TCompletionToken>(handler), std::forward<TArgs>(args)...));
}

template<typename TTokenType>
struct TFormatter {
    std::string operator()(const TConstArrayRef<const TTokenType>& arr) const {
        const auto str = boost::algorithm::join(
            arr | boost::adaptors::transformed(
                      [](auto&& token) {
                          return token.GetValue();
                      }),
            ", ");
        return "[" + str + "]";
    }
};

template <typename TFormatter, typename T>
void PrintChunks(std::stringstream& out, const TFormatter& fmt, const std::vector<NDiff::TChunk<T>>& chunks) {
    for (auto chunk = chunks.begin(); chunk != chunks.end(); ++chunk) {
        if (!chunk->Left.empty() || !chunk->Right.empty()) {
            out << "(";
            out << fmt(chunk->Left);
            out << "|";
            out << fmt(chunk->Right);
            out << ")";
        }
        if (chunk->Common.size()) {
            out << fmt(chunk->Common);
        }
    }
}

}
