#pragma once
#include <mail/template_master/lib/worker/db_pooler/db_pooler.h>
#include <mail/template_master/lib/worker/worker/worker.h>
#include <mail/template_master/lib/worker/db_pooler/config.h>
#include <mail/template_master/lib/types/context.h>
#include <mail/template_master/lib/utils/utils.h>
#include <mail/template_master/lib/types/asio.h>
#include <mail/template_master/lib/db/database_module.h>

#include <mail/yreflection/include/yamail/data/deserialization/ptree_reader.h>
#include <mail/yplatform/include/yplatform/module.h>
#include <mail/yplatform/include/yplatform/find.h>

#include <memory>

namespace NTemplateMaster::NDbPooler {

class TDbPooler : public IDatabasePooler, public yplatform::module {
private:
    decltype(auto) SharedFromThis() noexcept {
        return std::dynamic_pointer_cast<TDbPooler>(shared_from_this());
    }

    auto CreateContext() const noexcept {
        auto ctx = boost::make_shared<ymod_webserver::context>();
        auto context = boost::make_shared<TContext>(ctx, "TDbPooler");
        return context;
    }
public:
    TDbPooler(TReactor& reactor)
        : Reactor(reactor)
        , Strand(*Reactor.io())
        , TimerFetchTasks(Strand.get_io_context())
        , TimerFindTimeoutTasks(Strand.get_io_context())
    {}

    void init(const yplatform::ptree& config) {
        Config = yamail::data::deserialization::fromPtree<TConfig>(config);
    }

    void start() {
        boost::asio::spawn(Strand, [self=SharedFromThis()](TYield yield) {
            boost::system::error_code ec;
            while (ec != boost::asio::error::operation_aborted) {
                auto context = self->CreateContext();
                self->FetchTasks(context, yield);
                self->TimerFetchTasks.expires_from_now(boost::posix_time::milliseconds(self->Config.FetchTasksInterval.count()));
                self->TimerFetchTasks.async_wait(yield[ec]);
            }
        }, NUtils::kCoroutineAttributes);

        boost::asio::spawn(Strand, [self=SharedFromThis()](TYield yield) {
            boost::system::error_code ec;
            while (ec != boost::asio::error::operation_aborted) {
                auto context = self->CreateContext();
                self->FindTimeoutTasks(context, yield);
                self->TimerFindTimeoutTasks.expires_from_now(boost::posix_time::milliseconds(self->Config.FindTimeoutTasksInterval.count()));
                self->TimerFindTimeoutTasks.async_wait(yield[ec]);
            }
        }, NUtils::kCoroutineAttributes);
    }

    void stop() {
        TimerFetchTasks.cancel();
    }

private:
    void FetchTasks(TContextPtr context, TYield yield) noexcept {
        auto db = yplatform::find<NDatabase::IDatabase>("database");
        auto tasksExpected = db->GetReadyTasks(context, Config.TasksLimitLoad, yield);
        if (tasksExpected) {
            const auto tasks = tasksExpected.value();
            LOGDOG_(context->GetLogger(), notice, NTemplateMaster::NLog::tasks=tasks)
            auto worker = yplatform::find<NTemplateMaster::NWorker::IWorker>("worker");
            worker->RunTasks(context, tasks);
        } else {
            LOGDOG_(context->GetLogger(), error,
                    NTemplateMaster::NLog::error_code=tasksExpected.error())
        }
    }

    void FindTimeoutTasks(TContextPtr context, TYield yield) noexcept {
        auto db = yplatform::find<NDatabase::IDatabase>("database");
        auto res = db->MoveTimeoutTask(context, yield);
        if (!res) {
            LOGDOG_(context->GetLogger(), error,
                    NTemplateMaster::NLog::error_code=res.error())
        }
    }

private:
    TReactor& Reactor;
    TStrand Strand;
    TDeadlineTimer TimerFetchTasks;
    TDeadlineTimer TimerFindTimeoutTasks;
    TConfig Config;
};

}
