#pragma once
#include <mail/template_master/lib/template_master/template_master.h>
#include <mail/template_master/lib/db/database_module.h>
#include <mail/template_master/lib/worker/tasks/task.h>
#include <mail/template_master/lib/types/template/stable_sign.h>
#include <mail/template_master/lib/http/types.h>

#include <mail/yreflection/include/yamail/data/deserialization/yajl.h>
#include <mail/yreflection/include/yamail/data/reflection.h>
#include <mail/yplatform/include/yplatform/find.h>

#include <memory>
#include <iostream>

namespace NTemplateMaster::NWorker::NTask {


class TIndexTemplateFeaturesContext {
public:
    TTemplateStableSign StableSign;
};


class TIndexTemplateFeaturesTask : public ITask {
private:
    using ITemplateMaster = NTemplateMaster::NHttp::ITemplateMaster;
    using TContentProcessor = NTemplateMaster::NHttp::TContentProcessor;
    using TRequest = NTemplateMaster::THttpRouteRequest;
public:
    TIndexTemplateFeaturesTask(TTaskInfo info)
        : ITask(std::move(info))
    {
        yamail::data::deserialization::fromJson(TaskInfo.Context, Context);
    }

    virtual TTaskExecutionResult Execute(TContextPtr context, TYield yield) override {
        auto db = yplatform::find<NDatabase::IDatabase>("database");
        const auto dbTemplate = db->FindTemplateByStableSign(context, Context.StableSign, yield);
        if (!dbTemplate) {
            return {ETaskStatus::FAILED, {dbTemplate.error().message()}};
        }
        const auto templOpt = dbTemplate.value();
        if (!templOpt) {
            return {ETaskStatus::FAILED, {"Requested template not found"}};
        }
        auto contentProcessor = yplatform::find<ITemplateMaster>("template_master")->GetContentProcessor();
        const auto templStable = templOpt.value()->CreateStableTemplate<TContentProcessor, TRequest>(*contentProcessor);
        const auto digest = templStable->GetFeatures();
        LOGDOG_(context->GetLogger(), notice,
                NTemplateMaster::NLog::type="TIndexTemplateFeaturesTask",
                NTemplateMaster::NLog::stable_sign=templStable->GetStableSign(),
                NTemplateMaster::NLog::digest=digest);
        const auto indexResult = db->IndexTemplateFeatures(context, digest, templStable->GetStableSign(), yield);
        if (!indexResult) {
            return {ETaskStatus::FAILED, {indexResult.error().message()}};
        }
        return {ETaskStatus::SUCCESS, {}};
    }

    static constexpr auto Name() noexcept {
        return "index_template_features";
    }
private:
    TIndexTemplateFeaturesContext Context;
};

}

YREFLECTION_ADAPT_ADT(::NTemplateMaster::NWorker::NTask::TIndexTemplateFeaturesContext,
    YREFLECTION_MEMBER_RENAMED(NTemplateMaster::TTemplateStableSign, stable_sign, StableSign)
)
