CREATE TABLE sherlock.template_features (
    feature                 integer   primary key,
    stable_signs            bigint[]  NOT NULL,
    used                    boolean   NOT NULL DEFAULT TRUE,
    last_index_ts           timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX i_template_features_search
    ON sherlock.template_features USING gin (stable_signs) WITH (FASTUPDATE=OFF);

CREATE INDEX i_template_features_length
    ON sherlock.template_features (COALESCE(array_length(stable_signs, 1), 0));

CREATE INDEX i_template_features_feature_not_used
    ON sherlock.template_features (feature) WHERE not used;

CREATE INDEX i_template_features_last_index_ts
    ON sherlock.template_features (last_index_ts);

CREATE TABLE sherlock.template_meta (
    stable_sign                     bigint primary key,
    attributes                      jsonb NOT NULL DEFAULT '[]'::jsonb,
    features                        integer[] NOT NULL DEFAULT array[]::integer[],
    parent_templates                bigint[] NOT NULL DEFAULT array[]::bigint[],
    created                         timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP
);

ALTER TABLE template_meta ADD CONSTRAINT templates_meta_stable_sign
    FOREIGN KEY (stable_sign) REFERENCES template_bodies(stable_sign);

CREATE INDEX i_template_meta_features
    ON sherlock.template_meta USING gin (features gin__int_ops) WITH (FASTUPDATE=OFF);

CREATE TYPE sherlock.task_status AS ENUM (
    'NEW',
    'STARTED',
    'FAILED',
    'SUCCESS',
    'TIMEOUT'
);

CREATE TABLE sherlock.tasks (
    id              UUID                 PRIMARY KEY DEFAULT uuid_generate_v1(),
    type            TEXT                 NOT NULL,
    created         TIMESTAMP            NOT NULL DEFAULT current_timestamp,
    started         TIMESTAMP            DEFAULT NULL,
    schedule        TIMESTAMP            NOT NULL,
    timeout         INTERVAL             NOT NULL,
    status          sherlock.task_status NOT NULL DEFAULT 'NEW',
    context         jsonb                NOT NULL

    CONSTRAINT task_created_before_schedule CHECK (created <= schedule),
    CONSTRAINT task_created_before_started CHECK (started IS NULL OR (created <= started)),
    CONSTRAINT task_scheduled_before_started CHECK (started IS NULL OR (schedule <= started))
);

CREATE INDEX i_tasks_status_schedule
    ON sherlock.tasks (status, schedule);

CREATE INDEX i_task_type_status
    ON sherlock.tasks (type, status);

CREATE TABLE sherlock.recent_tasks
(
    id              UUID                 NOT NULL,
    type            TEXT                 NOT NULL,
    created         TIMESTAMPTZ          NOT NULL,
    started         TIMESTAMPTZ          NOT NULL,
    finished        TIMESTAMPTZ          NOT NULL DEFAULT current_timestamp,
    status          sherlock.task_status NOT NULL,
    error           TEXT                 DEFAULT NULL,
    context         jsonb
) PARTITION BY RANGE (finished);

CREATE TABLE sherlock.recent_tasks_template
(
    LIKE sherlock.recent_tasks INCLUDING DEFAULTS
);

ALTER TABLE sherlock.recent_tasks_template
    ADD CONSTRAINT recent_task_status_is_valid CHECK (
        status = 'FAILED' OR status = 'SUCCESS' OR status = 'TIMEOUT'
        ),
    ADD CONSTRAINT recent_task_started_is_present CHECK (started IS NOT NULL),
    ADD CONSTRAINT recent_task_finished_is_present CHECK (finished IS NOT NULL),
    ADD CONSTRAINT recent_task_started_before_finished CHECK (started < finished),
    ADD CONSTRAINT recent_task_failed_status_with_error CHECK (
            ((status = 'FAILED' OR status = 'TIMEOUT') AND error IS NOT NULL) OR
            (status = 'SUCCESS' AND error IS NULL)
        );

SELECT create_parent(
   p_parent_table := 'sherlock.recent_tasks',
   p_template_table := 'sherlock.recent_tasks_template',
   p_control := 'finished',
   p_type := 'native',
   p_interval := 'daily',
   p_premake := 7,
   p_jobmon := false
);

UPDATE part_config
SET retention            = '7 days',
    retention_keep_table = false,
    retention_keep_index = false
WHERE parent_table = 'sherlock.recent_tasks';
