#include <mail/template_master/lib/db/operations/delete_template.h>
#include <mail/template_master/ut/mock/connection_provider_mock.h>
#include <mail/template_master/ut/mock/query_repository_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>
#include <chrono>

namespace NTemplateMaster::NTests {

using namespace std::chrono_literals;
using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;

template<typename T>
using TDeleteTemplateOp = NTemplateMaster::NDatabase::Operations::TDeleteTemplateOp<T>;

class TDeleteTemplateTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TDeleteTemplateTest)
        UNIT_TEST(NotDeleted)
        UNIT_TEST(SuccessDeleted)
        UNIT_TEST(Error)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
    }

    void NotDeleted() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            TTemplateBody body(R"([["1", "2"], ["3"]])");
            TDeleteTemplateOp<decltype(queryRepo)> op(queryRepo, stableSign);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            std::vector<typename TDeleteTemplate::result_type> rows;
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_EQ(res.value(), static_cast<size_t>(0));
        });
    }

    void SuccessDeleted() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            TTemplateBody body(R"([["1", "2"], ["3"]])");
            TDeleteTemplateOp<decltype(queryRepo)> op(queryRepo, stableSign);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            std::vector<typename TDeleteTemplate::result_type> rows;
            rows.emplace_back(stableSign);
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_EQ(res.value(), static_cast<size_t>(1));
        });
    }

    void Error() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            TTemplateBody body(R"([["1", "2"], ["3"]])");
            TDeleteTemplateOp<decltype(queryRepo)> op(queryRepo, stableSign);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            std::vector<typename TDeleteTemplate::result_type> rows;
            rows.emplace_back(stableSign);
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            auto ec = boost::system::errc::make_error_code(boost::system::errc::not_supported);
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(ec));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_FALSE(res);
            EXPECT_EQ(res.error(), ec);
        });
    }
private:
    NTemplateMaster::TContextPtr Context;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TDeleteTemplateTest);
