#include <mail/template_master/lib/db/operations/find_similar_templates.h>
#include <mail/template_master/ut/mock/connection_provider_mock.h>
#include <mail/template_master/ut/mock/query_repository_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>
#include <chrono>

namespace NTemplateMaster::NTests {

using namespace std::chrono_literals;
using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;

template<typename T>
using TFindSimilarTemplatesOp = NTemplateMaster::NDatabase::Operations::TFindSimilarTemplatesOp<T>;

class TFindSimilarTemplatesTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TFindSimilarTemplatesTest)
        UNIT_TEST(NotFound)
        UNIT_TEST(Found)
        UNIT_TEST(Error)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
    }

    void NotFound() {
        Spawn([=]([[maybe_unused]]TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateFeaturesSet sign({1, 2, 3});
            TFindSimilarTemplatesOp<decltype(queryRepo)> op(queryRepo, sign, 3, 50);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*connProvider, Begin(_))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Execute(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            std::vector<typename TFindSimilarTemplates::result_type> rows;
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Commit(_))
                    .WillOnce(Return(boost::system::error_code()));
            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_EQ(res.value().size(), static_cast<size_t>(0));
        });
    }

    void Found() {
        Spawn([=]([[maybe_unused]]TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            TTemplateFeaturesVector sign2({3, 4, 5});
            ozo::pg::jsonb body(R"([[], []])");
            TFindSimilarTemplatesOp<decltype(queryRepo)> op(queryRepo, sign, 3, 50);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*connProvider, Begin(_))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Execute(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            std::vector<typename TFindSimilarTemplates::result_type> rows;
            rows.emplace_back(std::make_tuple(stableSign, sign2, body, 0.8));
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Commit(_))
                    .WillOnce(Return(boost::system::error_code()));
            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            const auto templates = res.value();
            EXPECT_EQ(templates.size(), static_cast<size_t>(1));
            EXPECT_EQ(templates.at(0)->GetStableSign(), stableSign);
            EXPECT_EQ(templates.at(0)->GetFeatures(), TTemplateFeaturesSet(sign2.begin(), sign2.end()));
            EXPECT_EQ(templates.at(0)->Json(), body.raw_string());
        });
    }

    void Error() {
        Spawn([=] ([[maybe_unused]]TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateFeaturesSet sign({1, 2, 3});
            TFindSimilarTemplatesOp<decltype(queryRepo)> op(queryRepo, sign, 3, 50);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*connProvider, Begin(_))
                    .WillOnce(Return(boost::system::error_code()));

            auto ec = boost::system::errc::make_error_code(boost::system::errc::not_supported);
            EXPECT_CALL(*connProvider, Execute(_, _))
                    .WillOnce(Return(ec));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_FALSE(res);
            EXPECT_EQ(res.error(), ec);
        });
    }

private:
    NTemplateMaster::TContextPtr Context;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TFindSimilarTemplatesTest);
