#include <mail/template_master/lib/db/operations/find_template_by_stable_sign.h>
#include <mail/template_master/ut/mock/connection_provider_mock.h>
#include <mail/template_master/ut/mock/query_repository_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>
#include <chrono>
namespace NTemplateMaster::NTests {

using namespace std::chrono_literals;
using ::testing::Return;
using ::testing::ReturnRef;
using ::testing::_;
using ::testing::InSequence;

template<typename T>
using TFindTemplateByStableSignOp = NTemplateMaster::NDatabase::Operations::TFindTemplateByStableSignOp<T>;

class TFindTemplateByStableSignTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TFindTemplateByStableSignTest)
        UNIT_TEST(NotFound)
        UNIT_TEST(Found)
        UNIT_TEST(Error)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
    }

    void NotFound() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            TTemplateBody body(R"([["1", "2"], ["3"]])");
            TFindTemplateByStableSignOp<decltype(queryRepo)> op(queryRepo, stableSign);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            std::vector<typename TFindTemplateByStableSign::result_type> rows;
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_FALSE(res.value().has_value());
        });
    }

    void Found() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesVector features({1, 2, 3});
            TTemplateBody body(R"([["1", "2"], ["3"]])");
            TFindTemplateByStableSignOp<decltype(queryRepo)> op(queryRepo, stableSign);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            std::vector<typename TFindTemplateByStableSign::result_type> rows;
            rows.emplace_back(std::make_tuple(stableSign, features, body));
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_TRUE(res.value().has_value());
            const auto templ = res.value().value();
            EXPECT_EQ(templ->GetStableSign(), stableSign);
            EXPECT_EQ(templ->GetFeatures(), TTemplateFeaturesSet(features.begin(), features.end()));
            EXPECT_EQ(templ->Json(), body.raw_string());
        });
    }

    void Error() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TFindTemplateByStableSignOp<decltype(queryRepo)> op(queryRepo, stableSign);

            EXPECT_CALL(*queryRepo.mock, make_query())
                    .WillOnce(Return(FakeQuery()));

            std::vector<typename TFindTemplateByStableSign::result_type> rows;
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            auto ec = boost::system::errc::make_error_code(boost::system::errc::not_supported);
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(ec));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_FALSE(res);
            EXPECT_EQ(res.error(), ec);
        });
    }
private:
    NTemplateMaster::TContextPtr Context;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TFindTemplateByStableSignTest);
