#include <mail/template_master/lib/db/operations/merge_templates.h>
#include <mail/template_master/ut/mock/connection_provider_mock.h>
#include <mail/template_master/ut/mock/query_repository_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>
#include <chrono>

namespace NTemplateMaster::NTests {

using namespace std::chrono_literals;
using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;

template<typename T>
using TMergeTemplatesOp = NTemplateMaster::NDatabase::Operations::TMergeTemplatesOp<T>;

class TMergeTemplatesTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TMergeTemplatesTest)
        UNIT_TEST(NotAllSearchFlagsUpdated)
        UNIT_TEST(SuccessUpdateSearchFlagsAndSaveNewTemplate)
        UNIT_TEST(SuccessUpdateSearchFlagsAndSaveExistsTemplate)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
    }

    void NotAllSearchFlagsUpdated() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            std::string body(R"([["1", "2"], ["3"]])");
            TDatabaseTemplatePtr templ = std::make_shared<TDatabaseTemplate>(stableSign, sign, body, TJsonAttributesArray{});
            TMergeTemplatesOp<decltype(queryRepo)> op(queryRepo, templ,  50, {789, 456});

            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*connProvider, Begin(_))
                    .WillOnce(Return(boost::system::error_code()));

            std::vector<typename TSelectForUpdateSearchFlag::result_type> rows;
            rows.emplace_back(stableSign);
            EXPECT_CALL(*connProvider, GetOut())
                    .WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Rollback(_))
                    .WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_FALSE(res.value());
        });
    }

    void SuccessUpdateSearchFlagsAndSaveNewTemplate() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            std::string body(R"([["1", "2"], ["3"]])");
            TDatabaseTemplatePtr templ = std::make_shared<TDatabaseTemplate>
                    (stableSign, sign, body, TJsonAttributesArray{});
            TMergeTemplatesOp<decltype(queryRepo)> op(queryRepo, templ,  50, {789, 456});

            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*connProvider, Begin(_)).WillOnce(Return(boost::system::error_code()));

            std::vector<typename TSelectForUpdateSearchFlag::result_type> rows;
            rows.emplace_back(stableSign);
            rows.emplace_back(stableSign);
            EXPECT_CALL(*connProvider, GetOut()).WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _)).WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Execute(_, _)).WillOnce(Return(boost::system::error_code()));
            std::vector<typename TSaveTemplate::result_type> rowsSave;
            rowsSave.emplace_back(stableSign);
            EXPECT_CALL(*connProvider, GetOut()).WillOnce(Return(rowsSave));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Commit(_)).WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_TRUE(res.value());
        });
    }

    void SuccessUpdateSearchFlagsAndSaveExistsTemplate() {
        Spawn([=](TYield yield) {
            const InSequence s;
            TQueryRepository queryRepo;
            queryRepo.mock = GetQueryRepositoryMock();
            auto connProvider = GetConnectionProviderMock();
            TTemplateStableSign stableSign(123);
            TTemplateFeaturesSet sign({1, 2, 3});
            std::string body(R"([["1", "2"], ["3"]])");
            TDatabaseTemplatePtr templ = std::make_shared<TDatabaseTemplate>
                    (stableSign, sign, body, TJsonAttributesArray{});
            TMergeTemplatesOp<decltype(queryRepo)> op(queryRepo, templ,  50, {789, 456});

            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));

            EXPECT_CALL(*connProvider, Begin(_)).WillOnce(Return(boost::system::error_code()));

            std::vector<typename TSelectForUpdateSearchFlag::result_type> rows;
            rows.emplace_back(stableSign);
            rows.emplace_back(stableSign);
            EXPECT_CALL(*connProvider, GetOut()).WillOnce(Return(rows));
            EXPECT_CALL(*connProvider, Request(_, _)).WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Execute(_, _)).WillOnce(Return(boost::system::error_code()));
            std::vector<typename TSaveTemplate::result_type> rowsSave;
            EXPECT_CALL(*connProvider, GetOut()).WillOnce(Return(rowsSave));
            EXPECT_CALL(*connProvider, Request(_, _))
                    .WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*queryRepo.mock, make_query()).WillOnce(Return(FakeQuery()));
            EXPECT_CALL(*connProvider, Execute(_, _)).WillOnce(Return(boost::system::error_code()));

            EXPECT_CALL(*connProvider, Commit(_)).WillOnce(Return(boost::system::error_code()));

            auto res = op(Context, connProvider, 1s, yield);
            EXPECT_TRUE(res);
            EXPECT_TRUE(res.value());
        });
    }
private:
    NTemplateMaster::TContextPtr Context;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TMergeTemplatesTest);
