#pragma once
#include <mail/template_master/lib/db/queries/query.h>
#include <mail/template_master/lib/types/expected.h>
#include <mail/template_master/lib/types/asio.h>
#include <mail/template_master/lib/db/operations/operation_traits.h>

#include <mail/template_master/ut/mock/query_repository_mock.h>

#include <ozo/error.h>
#include <ozo/time_traits.h>
#include <ozo/shortcuts.h>

#include <library/cpp/testing/gmock_in_unittest/gmock.h>

#include <type_traits>
#include <any>

namespace NTemplateMaster::NTests {

using ::testing::StrictMock;
using namespace NDatabase::NQuery;

template<typename TQuery>
using TResult = std::vector<typename TQuery::result_type>;

template<typename TQuery>
using TOut = decltype(ozo::into(std::declval<TResult<TQuery> &>()));

class TConnectionProviderMock;

using TConnectionProviderMockPtr = std::shared_ptr<StrictMock<TConnectionProviderMock>>;

class TConnectionProviderMock {
public:
    MOCK_METHOD(ozo::error_code, Request, (
            const FakeQuery&,
            const ozo::time_traits::duration&), (const));

    MOCK_METHOD(ozo::error_code, Execute, (
            const FakeQuery&,
            const ozo::time_traits::duration&), (const));

    MOCK_METHOD(ozo::error_code, Begin, (const ozo::time_traits::duration&), (const));
    MOCK_METHOD(ozo::error_code, Commit, (const ozo::time_traits::duration&), (const));
    MOCK_METHOD(ozo::error_code, Rollback, (const ozo::time_traits::duration&), (const));
    MOCK_METHOD(std::any, GetOut, (), (const));

    friend std::string get_error_context(TConnectionProviderMockPtr) {
        return "dummy";
    }

    friend std::string_view error_message(TConnectionProviderMockPtr) {
        return "dummy";
    }
};

using TConnectionProviderMockPtr = std::shared_ptr<StrictMock<TConnectionProviderMock>>;

inline auto GetConnectionProviderMock() noexcept {
    return std::make_shared<StrictMock<TConnectionProviderMock>>();
}

}

namespace NTemplateMaster::NDatabase::Operations::Adl {

template <>
struct TRequestImpl<NTemplateMaster::NTests::TConnectionProviderMockPtr> {
    template <typename TConnectionProvider, typename TQuery, typename TOut>
    static auto Apply(
            TConnectionProvider&& self,
            const TQuery& q,
            const ozo::time_traits::duration& timeout,
            TOut out,
            TYield yield)
    {
        auto outMock = std::forward<TConnectionProvider>(self)->GetOut();
        auto rows = std::any_cast<typename TOut::container_type>(outMock);
        std::copy(rows.begin(), rows.end(), out);
        *yield.ec_ = std::forward<TConnectionProvider>(self)->Request(q, timeout);
        return self;
    }
};

template <>
struct TExecuteImpl<NTemplateMaster::NTests::TConnectionProviderMockPtr> {
    template <typename TConnectionProvider, typename TQuery>
    static auto Apply(
            TConnectionProvider&& self,
            const TQuery& q,
            const ozo::time_traits::duration& timeout,
            TYield yield)
    {
        *yield.ec_ = std::forward<TConnectionProvider>(self)->Execute(q, timeout);
        return self;
    }
};

template <>
struct TBeginImpl<NTemplateMaster::NTests::TConnectionProviderMockPtr> {
    template <typename TConnectionProvider>
    static auto Apply(TConnectionProvider&& self,
            const ozo::time_traits::duration& timeout,
            TYield yield) {
        *yield.ec_ = std::forward<TConnectionProvider>(self)->Begin(timeout);
        return self;
    }
};

template <>
struct TCommitImpl<NTemplateMaster::NTests::TConnectionProviderMockPtr> {
    template <typename TConnectionProvider>
    static auto Apply(TConnectionProvider&& self,
            const ozo::time_traits::duration& timeout,
            TYield yield) {
        *yield.ec_ = std::forward<TConnectionProvider>(self)->Commit(timeout);
        return self;
    }
};

template <>
struct TRollbackImpl<NTemplateMaster::NTests::TConnectionProviderMockPtr> {
    template <typename TConnectionProvider>
    static auto Apply(TConnectionProvider&& self,
            const ozo::time_traits::duration& timeout,
            TYield yield) {
        *yield.ec_ = std::forward<TConnectionProvider>(self)->Rollback(timeout);
        return self;
    }
};

}
