#include <mail/template_master/lib/template_master/handlers/find_template_handler.h>
#include <mail/template_master/ut/mock/db_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>

namespace NTemplateMaster::NTests {

using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;
using TFindTemplateHandler = NTemplateMaster::NHandlers::TFindTemplateHandler<TContentProcessor, TRequest>;

class TFindTemplateHandlerTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TFindTemplateHandlerTest)
        UNIT_TEST(TemplateFound)
        UNIT_TEST(NotFound)
        UNIT_TEST(Error)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
        ContentProcessor = CreateContentProcessor(3);
    }

    void TemplateFound() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto dbMock = GetDbMock();
            const TTemplateStableSign stableSign = 123;
            const TTemplateFeaturesSet sign({});
            const std::string json(R"([["1", "2"], ["3"]])");
            auto dbTemplate = std::make_shared<TDatabaseTemplate>(stableSign, sign, json, TJsonAttributesArray{});

            TFindTemplateHandler findTemplateHandler(Context, stableSign, *ContentProcessor, dbMock);
            EXPECT_CALL(*dbMock, FindTemplateByStableSign(Context, stableSign, _))
                .WillOnce(Return(TExpected<TOptional<TDatabaseTemplatePtr>>(dbTemplate)));

            const auto result = findTemplateHandler(yield);
            EXPECT_EQ(result.Status, EFindTemplateStatus::Found);
            EXPECT_EQ(result.StableSign, stableSign);
            EXPECT_TRUE(result.Chunks.has_value());
            EXPECT_EQ(result.Chunks.value(), TTemplateChunks<TTokenType>({{"1", "2"}, {"3"}}));
        });
    }

    void NotFound() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto dbMock = GetDbMock();
            const TTemplateStableSign stableSign = 123;
            const TTemplateFeaturesSet sign({});
            const std::string json(R"([["1", "2"], ["3"]])");
            auto dbTemplate = std::make_shared<TDatabaseTemplate>(stableSign, sign, json, TJsonAttributesArray{});

            TFindTemplateHandler findTemplateHandler(Context, stableSign, *ContentProcessor, dbMock);
            EXPECT_CALL(*dbMock, FindTemplateByStableSign(Context, stableSign, _))
                    .WillOnce(Return(TExpected<TOptional<TDatabaseTemplatePtr>>()));

            const auto result = findTemplateHandler(yield);
            EXPECT_EQ(result.Status, EFindTemplateStatus::NotFound);
        });
    }

    void Error() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto dbMock = GetDbMock();
            const TTemplateStableSign stableSign = 123;
            const TTemplateFeaturesSet sign({});
            const std::string json(R"([["1", "2"], ["3"]])");
            auto dbTemplate = std::make_shared<TDatabaseTemplate>(stableSign, sign, json, TJsonAttributesArray{});

            TFindTemplateHandler findTemplateHandler(Context, stableSign, *ContentProcessor, dbMock);
            auto ec = boost::system::errc::make_error_code(boost::system::errc::not_supported);
            EXPECT_CALL(*dbMock, FindTemplateByStableSign(Context, stableSign, _))
                    .WillOnce(Return(yamail::make_unexpected(ec)));

            const auto result = findTemplateHandler(yield);
            EXPECT_EQ(result.Status, EFindTemplateStatus::Error);
        });
    }
private:
    NTemplateMaster::TContextPtr Context;
    TContentProcessorPtr ContentProcessor;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TFindTemplateHandlerTest);
