#include <mail/template_master/lib/template_master/operations/merge_templates.h>
#include <mail/template_master/ut/mock/db_mock.h>
#include <mail/template_master/ut/mock/template_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>

namespace NTemplateMaster::NTests {

using ::testing::_;
using ::testing::InSequence;
using ::testing::SaveArg;

using TMergeSimilarTemplates = ::NTemplateMaster::TMergeSimilarTemplates<TContentProcessor, TRequest>;

class TMergeSimilarTemplatesTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TMergeSimilarTemplatesTest)
        UNIT_TEST(TryMergeTemplatesTwoTemplatesSuccess)
        UNIT_TEST(NotMergeSimilarityLowerThanSimilarityThreshold)
        UNIT_TEST(TryMergeTemplatesWithEternal)
        UNIT_TEST(NotMergeTwoEternal)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
        ContentProcessor = CreateContentProcessor(3);
    }

    void TryMergeTemplatesTwoTemplatesSuccess() {
        Spawn([=](TYield yield) {
            const InSequence s;
            std::string json1(R"([["1", "2"], ["3", "4", "5"], ["6"]])");
            TStableTemplatePtr templFirst = std::make_shared<TStableTemplate>(*ContentProcessor, json1, false);
            std::string json2(R"([["1", "2"], ["3", "4"], ["6"], ["8"]])");
            TStableTemplatePtr templSecond = std::make_shared<TStableTemplate>(*ContentProcessor, json2, false);

            TDatabaseTemplatePtr dbTemplate;
            auto dbMock = GetDbMock();
            EXPECT_CALL(*dbMock, AsyncMergeTemplates(_, _, _, _, _))
                    .WillOnce(SaveArg<1>(&dbTemplate));
            TMergeSimilarTemplates op(dbMock, *ContentProcessor, 0.6, 0.8, 5);
            auto result = op.TryMergeTemplates(Context, templFirst, templSecond, yield);
            EXPECT_TRUE(result);
            EXPECT_EQ(dbTemplate->Json(), R"([["1","2"],["3","4"],["6"]])");
        });
    }

    void NotMergeSimilarityLowerThanSimilarityThreshold() {
        Spawn([=](TYield yield) {
            const InSequence s;
            std::string json1(R"([["1", "2"], ["9", "4", "5"], ["6"]])");
            TStableTemplatePtr templFirst = std::make_shared<TStableTemplate>(*ContentProcessor, json1, false);
            std::string json2(R"([["1", "2"], ["3", "4"], ["6"], ["8"]])");
            TStableTemplatePtr templSecond = std::make_shared<TStableTemplate>(*ContentProcessor, json2, false);

            auto dbMock = GetDbMock();
            TMergeSimilarTemplates op(dbMock, *ContentProcessor, 0.6, 0.8, 5);
            auto result = op.TryMergeTemplates(Context, templFirst, templSecond, yield);
            EXPECT_FALSE(result);
        });
    }

    void TryMergeTemplatesWithEternal() {
        Spawn([=](TYield yield) {
            const InSequence s;
            std::string json1(R"([["1", "2"], ["3", "4", "5"], ["6"]])");
            TStableTemplatePtr templFirst = std::make_shared<TStableTemplate>(*ContentProcessor, json1, true);
            std::string json2(R"([["1", "2"], ["3", "4"], ["6"], ["8"]])");
            TStableTemplatePtr templSecond = std::make_shared<TStableTemplate>(*ContentProcessor, json2, false);

            auto dbMock = GetDbMock();
            EXPECT_CALL(*dbMock, UpdateSearchFlag(
                    _, false, TTemplatesStableSigns({templSecond->GetStableSign()}), _)).Times(1);
            TMergeSimilarTemplates op(dbMock, *ContentProcessor, 0.6, 0.8, 5);
            auto result = op.TryMergeTemplates(Context, templFirst, templSecond, yield);
            EXPECT_TRUE(result);
        });
    }

    void NotMergeTwoEternal() {
        Spawn([=](TYield yield) {
            const InSequence s;
            std::string json1(R"([["1", "2"], ["3", "4", "5"], ["6"]])");
            TStableTemplatePtr templFirst = std::make_shared<TStableTemplate>(*ContentProcessor, json1, true);
            std::string json2(R"([["1", "2"], ["3", "4"], ["6"], ["8"]])");
            TStableTemplatePtr templSecond = std::make_shared<TStableTemplate>(*ContentProcessor, json2, true);

            auto dbMock = GetDbMock();
            TMergeSimilarTemplates op(dbMock, *ContentProcessor, 0.6, 0.8, 5);
            auto result = op.TryMergeTemplates(Context, templFirst, templSecond, yield);
            EXPECT_FALSE(result);
        });
    }

private:
    NTemplateMaster::TContextPtr Context;
    TContentProcessorPtr ContentProcessor;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TMergeSimilarTemplatesTest);
