#include <mail/template_master/lib/template_master/handlers/route_template_handler.h>
#include <mail/template_master/ut/mock/template_master_mock.h>
#include <mail/template_master/ut/mock/router_mock.h>
#include <mail/template_master/ut/mock/template_pool_mock.h>
#include <mail/template_master/ut/mock/template_mock.h>
#include <mail/template_master/ut/environment.h>
#include <mail/template_master/ut/utils.h>

#include <mail/yreflection/include/yamail/data/serialization/yajl.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>

namespace NTemplateMaster::NTests {

using ::testing::Return;
using ::testing::ReturnRef;
using ::testing::_;
using ::testing::InSequence;
using TRouteTemplateHandler = NTemplateMaster::NHandlers::TRouteTemplateHandler<TMockTemplatePtr>;

class TRouteHandlerTest : public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TRouteHandlerTest)
        UNIT_TEST(DbError)
        UNIT_TEST(FoundInDb)
        UNIT_TEST(RouteRequestAndGetSuccessAnswer)
        UNIT_TEST(RouteRequestAndGetErrorAnswer)
        UNIT_TEST(RouteRequestWithDbHints)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        TTestsEnvironment::SetUp();
        Io = std::make_unique<boost::asio::io_context>();
        Context = GetContext();
        ContentProcessor = CreateContentProcessor(3);
    }

    void DbError() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto templateMasterMock = GetTemplateMasterMock();
            auto msg = GetTemplateMock();
            auto router = GetRouterMock();
            std::string body;

            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(*ContentProcessor));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(TTemplateFeaturesSet{}));
            EXPECT_CALL(*msg, Size()).WillOnce(Return(1));
            auto ec = boost::system::errc::make_error_code(boost::system::errc::not_supported);
            EXPECT_CALL(*templateMasterMock, FindSimilarTemplates(_, _, _))
                .WillOnce(Return(yamail::make_unexpected(ec)));

            TRouteTemplateHandler routeHandler(Context, body, msg, templateMasterMock, router);
            const auto result = routeHandler(yield);
            EXPECT_EQ(result.Status, EDetempleStatus::Error);
        });
    }

    void FoundInDb() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto templateMasterMock = GetTemplateMasterMock();
            auto msg = GetTemplateMock();
            auto msgTokens = BuildTokenSequence("1", "2", "5", "3", "4");
            auto router = GetRouterMock();
            std::string body;

            TTemplateStableSign stableSign = 584845343253660892;
            TTemplateFeaturesSet sign({1, 2, 3});
            std::string json(R"([["1", "2"], ["3", "4"]])");
            auto templ1 = std::make_shared<TDatabaseTemplate>(stableSign, sign, json, TJsonAttributesArray{});
            TExpected<TDatabaseTemplates> dbTemplates({templ1});

            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(*ContentProcessor));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(TTemplateFeaturesSet{}));
            EXPECT_CALL(*msg, Size()).WillOnce(Return(1));
            EXPECT_CALL(*templateMasterMock, FindSimilarTemplates(Context, _, _)).WillOnce(Return(dbTemplates));
            EXPECT_CALL(*templateMasterMock, MergeTemplates(_, _)).WillOnce(Return());
            EXPECT_CALL(*msg, GetTokens()).WillOnce(ReturnRef(msgTokens));

            TRouteTemplateHandler routeHandler(Context, body, msg, templateMasterMock, router);
            const auto result = routeHandler(yield);
            std::vector<std::vector<std::string>> delta({{}, {"5"}, {}});
            EXPECT_EQ(result.Status, EDetempleStatus::FoundInDb);
            EXPECT_EQ(result.StableSign.value(), stableSign);
            EXPECT_EQ(result.Delta.value(), delta);
        });
    }

    void RouteRequestAndGetSuccessAnswer() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto templateMasterMock = GetTemplateMasterMock();
            auto msg = GetTemplateMock();
            auto router = GetRouterMock();
            TExpected<TDatabaseTemplates> dbTemplates;
            THttpRouteRequest routeRequest{"<!doctype html><meta charset=utf-8><title>html</title>", "from@ya.ru", "subject", "queueid", {"uid"}};
            std::string body = yamail::data::serialization::toJson(routeRequest).str();
            THttpDetempleRequest detempleRequest{routeRequest, TDbHints{}};
            std::string detempleRequestBody = yamail::data::serialization::toJson(detempleRequest).str();
            TExpected<THttpResponse> routerResponse(
                    THttpResponse{200, {}, R"({"status":"FoundNotReadyInPool","delta":[[],[]],"attributes":[]})", ""});

            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(*ContentProcessor));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(TTemplateFeaturesSet{}));
            EXPECT_CALL(*msg, Size()).WillOnce(Return(1));
            EXPECT_CALL(*templateMasterMock, FindSimilarTemplates(Context, _, _)).WillOnce(Return(dbTemplates));
            EXPECT_CALL(*templateMasterMock, MergeTemplates(_, _)).WillOnce(Return());
            EXPECT_CALL(*msg, GetContent()).WillOnce(Return(routeRequest));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(TTemplateFeaturesSet{}));
            EXPECT_CALL(*router, SendRequest(Context, detempleRequestBody, _, _)).WillOnce(Return(routerResponse));

            TRouteTemplateHandler routeHandler(Context, body, msg, templateMasterMock, router);
            const auto result = routeHandler(yield);
            std::vector<std::vector<std::string>> delta({{}, {}});
            EXPECT_EQ(result.Status, EDetempleStatus::FoundNotReadyInPool);
            EXPECT_FALSE(result.StableSign.has_value());
            EXPECT_EQ(result.Delta.value(), delta);
        });
    }

    void RouteRequestWithDbHints() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto templateMasterMock = GetTemplateMasterMock();
            auto msg = GetTemplateMock();
            auto router = GetRouterMock();

            TTemplateStableSign sign{100500};
            TTemplateFeaturesSet features{12345};
            std::string tokens{R"([["html"]])"};
            auto msgTokens = BuildTokenSequence("hello", "world");
            TJsonAttributesArray attributes{};
            TExpected<TDatabaseTemplates> dbTemplates{{std::make_shared<TDatabaseTemplate>(sign, features, tokens, attributes)}};
            TDbHints dbHints{TDbHint{sign, std::vector(features.begin(), features.end()), tokens, attributes.raw_string()}};

            THttpRouteRequest routeRequest{"<!doctype html><meta charset=utf-8><title>html</title>", "from@ya.ru", "subject", "queueid", {"uid"}};
            std::string body = yamail::data::serialization::toJson(routeRequest).str();
            THttpDetempleRequest detempleRequest{routeRequest, dbHints};
            std::string detempleRequestBody = yamail::data::serialization::toJson(detempleRequest).str();
            TExpected<THttpResponse> routerResponse(
                    THttpResponse{200, {}, R"({"status":"FoundInDbHints","delta":[[],[]],"attributes":[]})", ""});

            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(*ContentProcessor));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(features));
            EXPECT_CALL(*msg, Size()).WillOnce(Return(1));
            EXPECT_CALL(*templateMasterMock, FindSimilarTemplates(Context, _, _)).WillOnce(Return(dbTemplates));
            EXPECT_CALL(*templateMasterMock, MergeTemplates(_, _)).WillOnce(Return());
            EXPECT_CALL(*msg, GetTokens()).WillOnce(ReturnRef(msgTokens));
            EXPECT_CALL(*msg, GetContent()).WillOnce(Return(routeRequest));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(features));
            EXPECT_CALL(*router, SendRequest(Context, detempleRequestBody, _, _)).WillOnce(Return(routerResponse));

            TRouteTemplateHandler routeHandler(Context, body, msg, templateMasterMock, router);
            const auto result = routeHandler(yield);
            std::vector<std::vector<std::string>> delta({{}, {}});
            EXPECT_EQ(result.Status, EDetempleStatus::FoundInDbHints);
            EXPECT_FALSE(result.StableSign.has_value());
            EXPECT_EQ(result.Delta.value(), delta);
        });
    }

    void RouteRequestAndGetErrorAnswer() {
        Spawn([=](TYield yield) {
            const InSequence s;
            auto templateMasterMock = GetTemplateMasterMock();
            auto msg = GetTemplateMock();
            auto router = GetRouterMock();
            THttpRouteRequest routeRequest{"<!doctype html><meta charset=utf-8><title>html</title>", "from@ya.ru", "subject", "queueid", {"uid"}};
            std::string body = yamail::data::serialization::toJson(routeRequest).str();
            THttpDetempleRequest detempleRequest{routeRequest, TDbHints{}};
            std::string detempleRequestBody = yamail::data::serialization::toJson(detempleRequest).str();

            TExpected<TDatabaseTemplates> dbTemplates;

            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(*ContentProcessor));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(TTemplateFeaturesSet{}));
            EXPECT_CALL(*msg, Size()).WillOnce(Return(1));
            EXPECT_CALL(*templateMasterMock, FindSimilarTemplates(Context, _, _)).WillOnce(Return(dbTemplates));
            EXPECT_CALL(*templateMasterMock, MergeTemplates(_, _)).WillOnce(Return());
            EXPECT_CALL(*msg, GetContent()).WillOnce(Return(routeRequest));
            EXPECT_CALL(*msg, GetFeatures()).WillOnce(Return(TTemplateFeaturesSet{}));
            EXPECT_CALL(*router, SendRequest(Context, detempleRequestBody, _, _))
                .WillOnce(Return(yamail::make_unexpected(boost::system::error_code())));

            TRouteTemplateHandler routeHandler(Context, body, msg, templateMasterMock, router);
            const auto result = routeHandler(yield);
            std::vector<std::vector<std::string>> delta({{}, {}});
            EXPECT_EQ(result.Status, EDetempleStatus::Error);
            EXPECT_FALSE(result.StableSign.has_value());
            EXPECT_FALSE(result.Delta.has_value());
        });
    }

private:
    NTemplateMaster::TContextPtr Context;
    TContentProcessorPtr ContentProcessor;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TRouteHandlerTest);
