#include <mail/template_master/lib/template_master/content_processor/shinger_print.h>
#include <mail/template_master/lib/types/operation_request/email_http_request.h>
#include <mail/template_master/lib/types/token/token.h>
#include <mail/template_master/lib/template_master/content_processor/traits.h>

#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/gtest.h>

#include <memory>

namespace NTemplateMaster::NTests {

class TShingerPrintTest : public TTestBase {
    UNIT_TEST_SUITE(TShingerPrintTest)
        UNIT_TEST(CreateShinger)
        UNIT_TEST(TokenizeSimple)
        UNIT_TEST(TokinizeWithEqualTokensMoreThanThreshold)
        UNIT_TEST(TokinizeWithMultiplyEqualTokens)
        UNIT_TEST(CalculateStableSign)
        UNIT_TEST(CalculateStableSign2)
        UNIT_TEST(Digest)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        ContentProcessor = CreateContentProcessor(3);
    }

    void CreateShinger() {
        const std::string s1 = "abra";
        const std::string s2 = "cadabra";
        const std::string s3 = "\r\nПроблем с дисками в DSM не обнаружено.";
        ASSERT_EQ(ContentProcessor->CreateShinger(s1), 127015074);
        ASSERT_EQ(ContentProcessor->CreateShinger(s2), 54952402);
        ASSERT_EQ(ContentProcessor->CreateShinger(s3), 265083815);
    }

    void TokenizeSimple() {
        const TRequest s1{"one<test>two>", "", "", "", {}};
        const auto tokens = ContentProcessor->Tokenize(s1);
        ASSERT_EQ(tokens.size(), static_cast<size_t>(3));
        ASSERT_EQ(tokens.at(0).GetValue(), "one");
        ASSERT_EQ(tokens.at(1).GetValue(), "<test>");
        ASSERT_EQ(tokens.at(2).GetValue(), "two>");
    }

    void TokinizeWithEqualTokensMoreThanThreshold() {
        const NTemplateMaster::THttpRouteRequest s{R"(<html><body><p>Уважаемые покупатели!</p>
<p>Обращаем Ваше внимание, что сегодня в продажу на сайте поступили новые<strong>монеты </strong></p>
<p>В случае, если монету из списка нельзя добавить в корзину, это значит, что она уже куплена.</p><hr/>
<div>
<p>Интернет-магазин ZooCoin</p>
<p><a href="http://zoocoin.ru">http://zoocoin.ru</a></p>
</div></body></html>)", "", "", "", {}};
        const auto tokens = ContentProcessor->Tokenize(s);
        const auto tokensExpected = BuildTokenSequence("<html>", "<body>", "<p>", "Уважаемые покупатели!",
                "</p>", "\n", "<p>", "Обращаем Ваше внимание, что сегодня в продажу на сайте поступили новые",
                "<strong>", "монеты ", "</strong>", "</p>", "\n", "<p>",
                "В случае, если монету из списка нельзя добавить в корзину, это значит, что она уже куплена.",
                "</p>", "<hr/>", "\n", "<div>", "\n<p>", "Интернет-магазин ZooCoin", "</p>\n",
                "<p><a href=\"http://zoocoin.ru\">", "http://zoocoin.ru", "</a>", "</p>\n", "</div>", "</body>", "</html>");
        ASSERT_TRUE(Compare(tokens, tokensExpected));
    }

    void TokinizeWithMultiplyEqualTokens() {
        const TRequest s{R"(<html><body><html><html><p><body><a><html><p><html><body><html><html><a><p><body><html><p><body><a><html><p><html><body><html><p><a>)", "", "", "", {}};
        const auto tokens = ContentProcessor->Tokenize(s);
        const auto tokensExpected = BuildTokenSequence("<html>", "<body>", "<html>", "<html>", "<p>", "<body>",
                "<a>", "<html><p>", "<html><body>", "<html><html>", "<a>", "<p>", "<body>", "<html><p>", "<body><a>",
                "<html><p>", "<html><body>", "<html><p><a>");
        ASSERT_TRUE(Compare(tokens, tokensExpected));
    }

    void CalculateStableSign() {
        TStringTokenSequence tokens = BuildTokenSequence("<td width=\"600\" align=\"left\" style=\"min-width: 600px;\">",
                "<!-- внешний контейнер -->", "отписаться от уведомлений", "</td>",
                "\n\r\n", "</html>", "</body>");
        const auto stableSign = ContentProcessor->CalculateStableSign(tokens);
        ASSERT_EQ(stableSign, 405238235952885753);
    }

    void CalculateStableSign2() {
        TStringTokenSequence tokens = BuildTokenSequence("1", "2", "3", "4");
        const auto stableSign = ContentProcessor->CalculateStableSign(tokens);
        ASSERT_EQ(stableSign, 584845343253660892);
    }

    void Digest() {
        const auto tokens = BuildTokenSequence("a", "b", "c", "d", "d", "d");
        const auto features = ContentProcessor->CalculateDigest(tokens);
        ASSERT_EQ(features.size(), static_cast<size_t>(4));
        ASSERT_EQ(features.count(13375323), 1);
        ASSERT_EQ(features.count(78160015), 1);
        ASSERT_EQ(features.count(136805897), 1);
        ASSERT_EQ(features.count(154056191), 1);
    }
private:
    ::NTemplateMaster::THighfreqHashes HighfreqHashes;
    TContentProcessorPtr ContentProcessor;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TShingerPrintTest);
