#include <mail/template_master/lib/template_pool/operations/get_candidate_templates.h>
#include <mail/template_master/ut/utils.h>
#include <mail/template_master/ut/environment.h>
#include <mail/template_master/ut/mock/template_cache_mock.h>
#include <mail/template_master/ut/mock/template_mock.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>
#include <unordered_set>

namespace NTemplateMaster::NTests {

using TGetCandidateTemplatesOperation = NTemplateMaster::NTemplatePool::TGetCandidateTemplatesOperation
        <TTemplateCacheMockPtr, TMockTemplatePtr>;
using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;

class TGetCandidateTemplatesOpTest: public TTestBase {
    UNIT_TEST_SUITE(TGetCandidateTemplatesOpTest)
        UNIT_TEST(GetCandidateTemplates)
        UNIT_TEST(FilterTemplates)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Context = GetContext();
        Cache = GetTemplateCacheMock();
        TTestsEnvironment::SetUp();
    }

    void GetCandidateTemplates() {
        const auto templateMock1 = GetTemplateMock();
        const auto templateMock2 = GetTemplateMock();
        const auto templateMock3 = GetTemplateMock();
        const auto templateMock4 = GetTemplateMock();
        const auto msg = GetTemplateMock();
        EXPECT_CALL(*Cache, FindSimilarTemplates(msg)).WillOnce(
                Return(std::unordered_set<TTemplateUniqueId>({1, 2, 3, 4})));

        EXPECT_CALL(*msg, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7, 8, 9, 10}));

        EXPECT_CALL(*Cache, FindTemplateById(1)).WillOnce(Return(templateMock1));
        EXPECT_CALL(*templateMock1, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7, 8, 9, 10}));

        EXPECT_CALL(*Cache, FindTemplateById(2)).WillOnce(Return(templateMock2));
        EXPECT_CALL(*templateMock2, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7, 8, 9}));

        EXPECT_CALL(*Cache, FindTemplateById(3)).WillOnce(Return(templateMock3));
        EXPECT_CALL(*templateMock3, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7}));

        EXPECT_CALL(*Cache, FindTemplateById(4)).WillOnce(Return(templateMock4));
        EXPECT_CALL(*templateMock4, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5}));

        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));
        EXPECT_CALL(*templateMock3, GetUniqueId()).WillOnce(Return(3));

        size_t maxTemplatesAfterFilter = 5;
        TGetCandidateTemplatesOperation Op(Cache, maxTemplatesAfterFilter, 0.6, 3);
        const auto result = Op.GetCandidateTemplates(Context, msg);
        EXPECT_EQ(result.size(), static_cast<size_t>(3));
        EXPECT_EQ(result.at(0), templateMock1);
        EXPECT_EQ(result.at(1), templateMock2);
        EXPECT_EQ(result.at(2), templateMock3);

        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock1.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock2.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock3.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock4.get()));
    }

    void FilterTemplates() {
        const auto templateMock1 = GetTemplateMock();
        const auto templateMock2 = GetTemplateMock();
        const auto templateMock3 = GetTemplateMock();
        const auto templateMock4 = GetTemplateMock();
        const auto msg = GetTemplateMock();

        EXPECT_CALL(*msg, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7, 8, 9, 10}));

        EXPECT_CALL(*Cache, FindTemplateById(1)).WillOnce(Return(templateMock1));
        EXPECT_CALL(*templateMock1, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7, 8, 9, 10}));

        EXPECT_CALL(*Cache, FindTemplateById(2)).WillOnce(Return(templateMock2));
        EXPECT_CALL(*templateMock2, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7, 8, 9}));

        EXPECT_CALL(*Cache, FindTemplateById(3)).WillOnce(Return(templateMock3));
        EXPECT_CALL(*templateMock3, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5, 6, 7}));

        EXPECT_CALL(*Cache, FindTemplateById(4)).WillOnce(Return(templateMock4));
        EXPECT_CALL(*templateMock4, GetFeatures())
            .WillOnce(Return(TTemplateFeaturesSet{1, 2, 3, 4, 5}));

        size_t maxTemplatesAfterFilter = 2;
        TGetCandidateTemplatesOperation Op(Cache, maxTemplatesAfterFilter, 0.6, 3);
        std::unordered_set<TTemplateUniqueId> templateIds({1, 2, 3, 4});
        const auto result = Op.FilterTemplates(Context, templateIds, msg);
        EXPECT_EQ(result.size(), maxTemplatesAfterFilter);
        EXPECT_EQ(result.at(0), templateMock1);
        EXPECT_EQ(result.at(1), templateMock2);
    }

private:
    NTemplateMaster::TContextPtr Context;
    TTemplateCacheMockPtr Cache;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TGetCandidateTemplatesOpTest)
