#include <mail/template_master/lib/template_master/content_processor/shinger_print.h>
#include <mail/template_master/lib/types/operation_request/email_http_request.h>
#include <mail/template_master/lib/template_pool/cache/min_hash.h>
#include <mail/template_master/lib/template_master/content_processor/traits.h>
#include <mail/template_master/lib/types/template/unstable_template.h>
#include <mail/template_master/ut/mock/template_mock.h>

#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>

namespace NTemplateMaster::NTests {

using TMinHash = ::NTemplateMaster::NTemplatePool::TMinHash<TMockTemplatePtr, THasher>;
using THashIterator = std::vector<NTemplateMaster::TTemplateFeature>::const_iterator;
using THashType = NTemplateMaster::NTemplatePool::THasherReturnType<THasher, THashIterator>;
using TBucket = std::unordered_map<THashType, std::unordered_set<TTemplateUniqueId>>;
using ::testing::Return;
using ::testing::InSequence;

class TMinHashTest: public TTestBase {
    UNIT_TEST_SUITE(TMinHashTest)
        UNIT_TEST(CalculateMinHash)
        UNIT_TEST(InsertAndErase)
        UNIT_TEST(FindSimilarTemplates)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        MinHash = std::make_unique<TMinHash>(2, 2,
                TTemplateFeaturesVector{4848, 9399, 2823, 40}, THasher());
    }

    void CalculateMinHash() {
        const InSequence s;
        const auto templateMock = GetTemplateMock();
        EXPECT_CALL(*templateMock, GetFeatures()).WillOnce(
                Return(TTemplateFeaturesSet{293, 12993, 9949172}));
        const auto minHash = MinHash->CalculateMinHash(templateMock);
        ASSERT_EQ(minHash.size(), static_cast<size_t>(2));
        ASSERT_EQ(minHash[0], static_cast<THashType>(10827));
        ASSERT_EQ(minHash[1], static_cast<THashType>(2863));
    }

    void InsertAndErase() {
        const InSequence s;
        const auto templateMock1 = GetTemplateMock();
        const auto digest1 = TTemplateFeaturesSet{293, 12993, 9949172};

        const auto templateMock2 = GetTemplateMock();
        const auto digest2 = TTemplateFeaturesSet{2923, 8438829, 283838, 48488, 389238};

        const auto templateMock3 = GetTemplateMock();
        const auto digest3 = TTemplateFeaturesSet{9994, 4444, 19, 192, 2199129, 1929134, 495908};

        EXPECT_CALL(*templateMock1, GetFeatures()).WillOnce(Return(digest1));
        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        EXPECT_CALL(*templateMock2, GetFeatures()).WillOnce(Return(digest2));
        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));
        EXPECT_CALL(*templateMock3, GetFeatures()).WillOnce(Return(digest3));
        EXPECT_CALL(*templateMock3, GetUniqueId()).WillOnce(Return(3));

        MinHash->InsertTemplate(templateMock1);
        MinHash->InsertTemplate(templateMock2);
        MinHash->InsertTemplate(templateMock3);
        auto buckets = MinHash->GetBuckets();
        ASSERT_EQ(buckets.size(), static_cast<size_t>(2));
        TBucket bucket0({{10827, {1}}, {18807, {2}}, {1897, {3}}});
        ASSERT_EQ(buckets[0], bucket0);
        TBucket bucket1({{2863, {1}}, {2991, {2}}, {2895, {3}}});
        ASSERT_EQ(buckets[1], bucket1);

        EXPECT_CALL(*templateMock2, GetFeatures()).WillOnce(Return(digest2));
        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));

        MinHash->EraseTemplate(templateMock2);
        buckets = MinHash->GetBuckets();
        ASSERT_EQ(buckets.size(), static_cast<size_t>(2));
        TBucket bucket3({{10827, {1}}, {1897, {3}}});
        ASSERT_EQ(buckets[0], bucket3);
        TBucket bucket4({{2863, {1}}, {2895, {3}}});
        ASSERT_EQ(buckets[1], bucket4);

        const auto templateMock4 = GetTemplateMock();
        const auto digest4 = TTemplateFeaturesSet{1, 2};

        EXPECT_CALL(*templateMock4, GetFeatures()).WillOnce(Return(digest4));
        EXPECT_CALL(*templateMock4, GetUniqueId()).WillOnce(Return(3));

        MinHash->EraseTemplate(templateMock4);
        buckets = MinHash->GetBuckets();
        ASSERT_EQ(buckets.size(), static_cast<size_t>(2));
        ASSERT_EQ(buckets[0], bucket3);
        ASSERT_EQ(buckets[1], bucket4);
    }

    void FindSimilarTemplates() {
        const InSequence s;

        const auto templateMock1 = GetTemplateMock();
        const auto features1 = TTemplateFeaturesSet{1, 2, 3, 4, 5};

        const auto templateMock2 = GetTemplateMock();
        const auto features2 = TTemplateFeaturesSet{1, 2, 3, 5, 6};

        const auto templateMock3 = GetTemplateMock();
        const auto features3 = TTemplateFeaturesSet{1, 2, 6, 5, 3};

        EXPECT_CALL(*templateMock1, GetFeatures()).WillOnce(Return(features1));
        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        EXPECT_CALL(*templateMock2, GetFeatures()).WillOnce(Return(features2));
        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));
        EXPECT_CALL(*templateMock3, GetFeatures()).WillOnce(Return(features3));
        EXPECT_CALL(*templateMock3, GetUniqueId()).WillOnce(Return(3));

        MinHash->InsertTemplate(templateMock1);
        MinHash->InsertTemplate(templateMock2);
        MinHash->InsertTemplate(templateMock3);

        EXPECT_CALL(*templateMock1, GetFeatures()).WillOnce(Return(features1));
        EXPECT_CALL(*templateMock2, GetFeatures()).WillOnce(Return(features2));
        EXPECT_CALL(*templateMock3, GetFeatures()).WillOnce(Return(features3));

        auto similar = MinHash->FindSimilarTemplates(templateMock1);
        ASSERT_EQ(similar, std::unordered_set<TTemplateUniqueId>({1}));
        similar = MinHash->FindSimilarTemplates(templateMock2);
        ASSERT_EQ(similar, std::unordered_set<TTemplateUniqueId>({2, 3}));
        similar = MinHash->FindSimilarTemplates(templateMock3);
        ASSERT_EQ(similar, std::unordered_set<TTemplateUniqueId>({2, 3}));
    }

private:
    std::unique_ptr<TMinHash> MinHash;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TMinHashTest)
