#include <mail/template_master/lib/template_pool/cache/template_cache.h>
#include <mail/template_master/ut/mock/template_mock.h>
#include <mail/template_master/ut/mock/min_hash_mock.h>
#include <mail/template_master/ut/utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>

namespace NTemplateMaster::NTests {

using TTemplateCache = NTemplateMaster::NTemplatePool::TTemplateCache<decltype(GetTemplateMock()), TMinHashMockPtr>;
using TTemplateCachePtr = NTemplateMaster::NTemplatePool::TTemplateCachePtr<decltype(GetTemplateMock()), TMinHashMockPtr>;
using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;
using ::testing::StrictMock;

class TTemplateCacheTest : public TTestBase {
    UNIT_TEST_SUITE(TTemplateCacheTest)
        UNIT_TEST(FindTemplateById)
        UNIT_TEST(Promote)
        UNIT_TEST(UpdateTemplate)
        UNIT_TEST(Erase)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        MinHash = GetMinHashMock();
        TemplateCache = std::make_shared<TTemplateCache>(3, MinHash, 42 /*similarTemplatesLimit*/);
    }

    void FindTemplateById() {
        const InSequence s;
        const auto templateMock1 = GetTemplateMock();
        const auto features1 = TTemplateFeaturesSet{22162015, 44192031, 107386536, 120536159, 142529758, 143303368, 153450037};

        const auto templateMock2 = GetTemplateMock();
        const auto features2 = TTemplateFeaturesSet{22162015, 44192031, 107386536, 143303368, 153450037};

        const auto templateMock3 = GetTemplateMock();
        const auto features3 = TTemplateFeaturesSet{9994, 4444, 22162015, 44192031, 107386536, 143303368, 153450037};

        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock1)).WillOnce(Return());

        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock2)).WillOnce(Return());

        EXPECT_CALL(*templateMock3, GetUniqueId()).WillOnce(Return(3));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock3)).WillOnce(Return());

        TemplateCache->InsertTemplate(templateMock1);
        TemplateCache->InsertTemplate(templateMock2);
        TemplateCache->InsertTemplate(templateMock3);
        ASSERT_EQ(TemplateCache->Size(), static_cast<size_t>(3));

        auto t = TemplateCache->FindTemplateById(2);
        ASSERT_EQ(t.value(), templateMock2);

        EXPECT_CALL(*MinHash, EraseTemplate(t.value())).WillOnce(Return());
        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));

        TemplateCache->EraseTemplate(t.value());
        ASSERT_EQ(TemplateCache->Size(), static_cast<size_t>(2));

        t = TemplateCache->FindTemplateById(2);
        ASSERT_FALSE(t.has_value());
        t = TemplateCache->FindTemplateById(3);
        ASSERT_EQ(t.value(), templateMock3);

        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock1.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock2.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock3.get()));
    }

    void Promote() {
        const InSequence s;
        const auto templateMock1 = GetTemplateMock();
        const auto templateMock2 = GetTemplateMock();
        const auto templateMock3 = GetTemplateMock();

        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock1)).WillOnce(Return());

        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock2)).WillOnce(Return());

        EXPECT_CALL(*templateMock3, GetUniqueId()).WillOnce(Return(3));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock3)).WillOnce(Return());

        TemplateCache->InsertTemplate(templateMock1);
        TemplateCache->InsertTemplate(templateMock2);
        TemplateCache->InsertTemplate(templateMock3);

        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        TemplateCache->PromoteTemplate(templateMock1);

        EXPECT_CALL(*templateMock1, GetUniqueId()).WillOnce(Return(1));
        TemplateCache->PromoteTemplate(templateMock1);

        const auto templateMock4 = GetTemplateMock();

        EXPECT_CALL(*MinHash, EraseTemplate(templateMock2)).WillOnce(Return());
        EXPECT_CALL(*templateMock2, GetUniqueId()).WillOnce(Return(2));

        EXPECT_CALL(*templateMock4, GetUniqueId()).WillOnce(Return(4));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock4)).WillOnce(Return());

        TemplateCache->InsertTemplate(templateMock4);
        ASSERT_EQ(TemplateCache->Size(), static_cast<size_t>(3));
        ASSERT_TRUE(TemplateCache->FindTemplateById(1).has_value());
        ASSERT_TRUE(TemplateCache->FindTemplateById(3).has_value());
        ASSERT_TRUE(TemplateCache->FindTemplateById(4).has_value());
        ASSERT_FALSE(TemplateCache->FindTemplateById(2).has_value());

        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock1.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock2.get()));
        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock3.get()));
    }

    void UpdateTemplate() {
        const InSequence s;
        const auto templateMock = GetTemplateMock();
        const TStringTokenSequence tokenSequence{TStringToken(), TStringToken("1"), TStringToken("2"),
                                                 TStringToken(), TStringToken("3"), TStringToken()};
        EXPECT_CALL(*MinHash, EraseTemplate(templateMock)).WillOnce(Return());
        EXPECT_CALL(*templateMock, UpdateTokens(_)).WillOnce(Return());
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock)).WillOnce(Return());
        TemplateCache->UpdateTemplate(templateMock, tokenSequence);

        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock.get()));
    }

    void Erase() {
        const InSequence s;
        const auto templateMock = GetTemplateMock();

        EXPECT_CALL(*templateMock, GetUniqueId()).WillOnce(Return(1));
        EXPECT_CALL(*MinHash, InsertTemplate(templateMock)).WillOnce(Return());

        TemplateCache->InsertTemplate(templateMock);
        auto res = TemplateCache->FindTemplateById(1);
        EXPECT_TRUE(res.has_value());
        EXPECT_EQ(res.value(), templateMock);

        EXPECT_CALL(*MinHash, EraseTemplate(templateMock)).WillOnce(Return());
        EXPECT_CALL(*templateMock, GetUniqueId()).WillOnce(Return(1));
        TemplateCache->EraseTemplate(templateMock);

        res = TemplateCache->FindTemplateById(1);
        EXPECT_FALSE(res.has_value());

        EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templateMock.get()));
    }

private:
    TMinHashMockPtr MinHash;
    TTemplateCachePtr TemplateCache;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TTemplateCacheTest);
