#include <mail/template_master/lib/template_pool/template_pool.h>
#include <mail/template_master/lib/template_pool/config.h>
#include <mail/template_master/ut/utils.h>
#include <mail/template_master/ut/environment.h>
#include <mail/template_master/ut/mock/template_pool_operations_mock.h>
#include <mail/template_master/ut/mock/db_mock.h>
#include <mail/template_master/ut/mock/template_cache_mock.h>

#include <library/cpp/testing/unittest/registar.h>

#include <memory>

namespace NTemplateMaster::NTests {

using ::testing::Return;
using ::testing::_;
using ::testing::InSequence;
using ::testing::StrictMock;
using TPool = NTemplatePool::TTemplatePool<
        TContentProcessor,
        TRequest,
        TMockTemplatePtr,
        TTemplateCacheMockPtr,
        TMockTemplatePtr,
        TCalculateBestMatchOperationMockPtr,
        TSaveReadyTemplateOperationMockPtr,
        TGetCandidateTemplatesOperationMockPtr
        >;

class TTemplatePoolTest: public TTestBase, public TWithSpawn {
    UNIT_TEST_SUITE(TTemplatePoolTest)
        UNIT_TEST(NoCandidateTemplatesFound)
        UNIT_TEST(NoCandidateTemplatesFoundInPoolButDbHintsContainSimilarTemplate)
        UNIT_TEST(HaveCandidateTemplatesButNoBestMatch)
        UNIT_TEST(NoBestMatchInPoolButDbHintsContainSimilarTemplate)
        UNIT_TEST(HaveBestMatchButTemplateNotReady)
        UNIT_TEST(HaveBestMatchAndSaveTemplate)
    UNIT_TEST_SUITE_END();
public:
    void SetUp() override {
        Context = GetContext();
        TTestsEnvironment::SetUp();
        GetCandidateTemplatesOp = GetGetCandidateTemplatesOperationMock();
        CalculateBestMatchOp = GetCalculateBestMatchOperationMock();
        SaveReadyTemplateOp = GetSaveReadyTemplateOperationMock();
        Cache = GetTemplateCacheMock();
        Io = std::make_unique<boost::asio::io_context>();
    }

    void NoCandidateTemplatesFound() {
        Spawn([&] (boost::asio::yield_context yield) {
            const InSequence s;
            NTemplateMaster::NTemplatePool::TConfig conf;
            conf.MatchesForReadyTemplate = 2;
            auto pool = std::make_shared<TPool>(Io.get(), conf, 1, CalculateBestMatchOp,
                    SaveReadyTemplateOp, Cache, GetCandidateTemplatesOp);
            const auto msg = GetTemplateMock();
            const auto templ = GetTemplateMock();
            const auto candidatesFromPool = std::vector<TMockTemplatePtr>{};

            EXPECT_CALL(*GetCandidateTemplatesOp, GetCandidateTemplates(Context, msg))
                    .WillOnce(Return(candidatesFromPool));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, candidatesFromPool))
                    .WillOnce(Return(std::nullopt));
            EXPECT_CALL(*msg, CreateUnstableTemplate()).WillOnce(Return(templ));
            EXPECT_CALL(*Cache, InsertTemplate(templ)).WillOnce(Return());
            auto result = pool->AddToPool(Context, msg, std::nullopt, yield);
            EXPECT_EQ(result.GetStatus(), EDetempleStatus::NotFound);

            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templ.get()));
        });
    }

    void NoCandidateTemplatesFoundInPoolButDbHintsContainSimilarTemplate() {
        Spawn([&] (boost::asio::yield_context yield) {
            const InSequence s;
            NTemplateMaster::NTemplatePool::TConfig conf;
            conf.MatchesForReadyTemplate = 2;
            auto pool = std::make_shared<TPool>(Io.get(), conf, 1, CalculateBestMatchOp,
                    SaveReadyTemplateOp, Cache, GetCandidateTemplatesOp);
            const auto msg = GetTemplateMock();
            const auto templ = GetTemplateMock();
            const auto candidatesFromPool = std::vector<TMockTemplatePtr>({});

            TTemplateStableSign sign{100500};
            TTemplateFeaturesSet features{12345};
            std::string tokens{R"([["html"]])"};
            auto msgTokens = BuildTokenSequence("hello", "world");
            TJsonAttributesArray attributes{R"([])"};
            auto dbTemplate = std::make_shared<TDatabaseTemplate>(sign, features, tokens, attributes);
            TExpected<TDatabaseTemplates> dbTemplates{{dbTemplate}};
            TDbHints dbHints{TDbHint{sign, std::vector(features.begin(), features.end()), tokens, attributes.raw_string()}};

            auto contentProcessor = TContentProcessor(1234567890);
            const auto matchesForReadyTemplate = 10;
            auto adaptedDbTemplate = std::make_shared<TAdaptedTemplate::element_type>(*dbTemplate, contentProcessor, matchesForReadyTemplate);
            EXPECT_CALL(*GetCandidateTemplatesOp, GetCandidateTemplates(Context, msg))
                    .WillOnce(Return(candidatesFromPool));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, candidatesFromPool))
                    .WillOnce(Return(std::nullopt));
            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(contentProcessor));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, testing::An<std::vector<decltype(adaptedDbTemplate)>>()))
                    .WillOnce(Return(TOptional<TMatchDbTemplate>({adaptedDbTemplate, TChunks<TTokenType>{}})));
            auto result = pool->AddToPool(Context, msg, dbHints, yield);
            EXPECT_EQ(result.GetStatus(), EDetempleStatus::FoundInDbHints);

            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templ.get()));
        });
    }

    void HaveCandidateTemplatesButNoBestMatch() {
        Spawn([&] (boost::asio::yield_context yield) {
            const InSequence s;
            NTemplateMaster::NTemplatePool::TConfig conf;
            conf.MatchesForReadyTemplate = 2;
            auto pool = std::make_shared<TPool>(Io.get(), conf, 1, CalculateBestMatchOp,
                    SaveReadyTemplateOp, Cache, GetCandidateTemplatesOp);
            const auto msg = GetTemplateMock();
            const auto templ = GetTemplateMock();
            std::vector<decltype(GetTemplateMock())> candidateTemplates({GetTemplateMock()});

            EXPECT_CALL(*GetCandidateTemplatesOp, GetCandidateTemplates(Context, msg))
                    .WillOnce(Return(candidateTemplates));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, candidateTemplates))
                    .WillOnce(Return(TOptional<TMatchTemplateMock>()));
            EXPECT_CALL(*msg, CreateUnstableTemplate()).WillOnce(Return(templ));
            EXPECT_CALL(*Cache, InsertTemplate(templ)).WillOnce(Return());
            auto result = pool->AddToPool(Context, msg, std::nullopt, yield);
            EXPECT_EQ(result.GetStatus(), EDetempleStatus::NotFound);

            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templ.get()));
        });
    }

    void NoBestMatchInPoolButDbHintsContainSimilarTemplate() {
        Spawn([&] (boost::asio::yield_context yield) {
            const InSequence s;
            NTemplateMaster::NTemplatePool::TConfig conf;
            conf.MatchesForReadyTemplate = 2;
            auto pool = std::make_shared<TPool>(Io.get(), conf, 1, CalculateBestMatchOp,
                    SaveReadyTemplateOp, Cache, GetCandidateTemplatesOp);
            const auto msg = GetTemplateMock();
            const auto templ = GetTemplateMock();
            std::vector<decltype(GetTemplateMock())> candidateTemplates({GetTemplateMock()});

            TTemplateStableSign sign{100500};
            TTemplateFeaturesSet features{12345};
            std::string tokens{R"([["html"]])"};
            const auto msgTokens = BuildTokenSequence("hello", "world");
            TJsonAttributesArray attributes{R"([])"};
            const auto dbTemplate = std::make_shared<TDatabaseTemplate>(sign, features, tokens, attributes);
            TExpected<TDatabaseTemplates> dbTemplates{{dbTemplate}};
            TDbHints dbHints{TDbHint{sign, std::vector(features.begin(), features.end()), tokens, attributes.raw_string()}};

            const auto contentProcessor = TContentProcessor(1234567890);
            const auto matchesForReadyTemplate = 10;
            auto adaptedDbTemplate = std::make_shared<TAdaptedTemplate::element_type>(*dbTemplate, contentProcessor, matchesForReadyTemplate);
            EXPECT_CALL(*GetCandidateTemplatesOp, GetCandidateTemplates(Context, msg))
                    .WillOnce(Return(candidateTemplates));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, candidateTemplates))
                    .WillOnce(Return(TOptional<TMatchTemplateMock>()));
            EXPECT_CALL(*msg, GetContentProcessor()).WillOnce(Return(contentProcessor));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, testing::An<std::vector<decltype(adaptedDbTemplate)>>()))
                    .WillOnce(Return(TOptional<TMatchDbTemplate>({adaptedDbTemplate, TChunks<TTokenType>{}})));
            auto result = pool->AddToPool(Context, msg, dbHints, yield);
            EXPECT_EQ(result.GetStatus(), EDetempleStatus::FoundInDbHints);

            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templ.get()));
        });
    }

    void HaveBestMatchButTemplateNotReady() {
        Spawn([&] (boost::asio::yield_context yield) {
            const InSequence s;
            NTemplateMaster::NTemplatePool::TConfig conf;
            conf.MatchesForReadyTemplate = 2;
            auto pool = std::make_shared<TPool>(Io.get(), conf, 1, CalculateBestMatchOp,
                    SaveReadyTemplateOp, Cache, GetCandidateTemplatesOp);
            const auto msg = GetTemplateMock();
            const auto templ = GetTemplateMock();
            std::vector<decltype(GetTemplateMock())> candidateTemplates({GetTemplateMock()});
            const TEmailAttributes attrs;
            TStringTokenSequence seqLeft({TTokenType("1")});
            NDiff::TChunk<TTokenType> chunk1;
            chunk1.Left = seqLeft;
            TChunks<TTokenType> diff({chunk1});
            TOptional<TMatchTemplateMock> matchResult({templ, diff});

            EXPECT_CALL(*GetCandidateTemplatesOp, GetCandidateTemplates(Context, msg))
                    .WillOnce(Return(candidateTemplates));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, candidateTemplates))
                    .WillOnce(Return(matchResult));

            EXPECT_CALL(*templ, GetHits()).WillOnce(Return(1));

            EXPECT_CALL(*Cache, PromoteTemplate(templ)).WillOnce(Return());
            EXPECT_CALL(*Cache, UpdateTemplate(templ, _)).WillOnce(Return());
            EXPECT_CALL(*msg, GetAttributes()).WillOnce(Return(attrs));
            EXPECT_CALL(*templ, PushAttributes(_)).WillOnce(Return());
            EXPECT_CALL(*templ, IncrementHits()).WillOnce(Return());

            EXPECT_CALL(*templ, GetAttributesArray()).WillOnce(Return(TAttributesArray<TEmailAttributes>({attrs})));
            auto result = pool->AddToPool(Context, msg, std::nullopt, yield);
            EXPECT_EQ(result.GetStatus(), EDetempleStatus::FoundNotReadyInPool);
            EXPECT_TRUE(result.Attributes.has_value());

            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templ.get()));
        });
    }

    void HaveBestMatchAndSaveTemplate() {
        Spawn([=](boost::asio::yield_context yield) {
            const InSequence s;

            NTemplateMaster::NTemplatePool::TConfig conf;
            conf.MatchesForReadyTemplate = 2;
            auto pool = std::make_shared<TPool>(Io.get(), conf, 1, CalculateBestMatchOp,
                    SaveReadyTemplateOp, Cache, GetCandidateTemplatesOp);
            const auto msg = GetTemplateMock();
            const auto templ = GetTemplateMock();
            std::vector<decltype(GetTemplateMock())> candidateTemplates({GetTemplateMock()});
            const TEmailAttributes attrs;
            TStringTokenSequence seqLeft({TTokenType("1")});
            NDiff::TChunk<TTokenType> chunk1;
            chunk1.Left = seqLeft;
            TChunks<TTokenType> diff({chunk1});

            TOptional<TMatchTemplateMock> matchResult({templ, diff});
            EXPECT_CALL(*GetCandidateTemplatesOp, GetCandidateTemplates(Context, msg))
                    .WillOnce(Return(candidateTemplates));
            EXPECT_CALL(*CalculateBestMatchOp, GetBestMatch(Context, msg, candidateTemplates))
                    .WillOnce(Return(matchResult));

            EXPECT_CALL(*templ, GetHits()).WillOnce(Return(3));

            EXPECT_CALL(*SaveReadyTemplateOp, Save(Context, templ, _)).WillOnce(Return());
            EXPECT_CALL(*templ, GetAttributesArray()).WillOnce(Return(TAttributesArray<TEmailAttributes>({attrs})));
            EXPECT_CALL(*templ, GetStableSign()).WillOnce(Return(1234));
            auto result = pool->AddToPool(Context, msg, std::nullopt, yield);
            EXPECT_EQ(result.GetStatus(), EDetempleStatus::FoundPreparedInPool);
            EXPECT_TRUE(result.Attributes.has_value());
            EXPECT_EQ(result.StableSign, 1234);

            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(msg.get()));
            EXPECT_TRUE(::testing::Mock::VerifyAndClearExpectations(templ.get()));
        });
    }
private:
    NTemplateMaster::TContextPtr Context;
    TGetCandidateTemplatesOperationMockPtr GetCandidateTemplatesOp;
    TCalculateBestMatchOperationMockPtr CalculateBestMatchOp;
    TSaveReadyTemplateOperationMockPtr SaveReadyTemplateOp;
    TTemplateCacheMockPtr Cache;
};

}

UNIT_TEST_SUITE_REGISTRATION(NTemplateMaster::NTests::TTemplatePoolTest)
