#!/usr/bin/python2
# -*- coding: UTF-8 -*-

import argparse
import json
import logging
import re
import requests
import time

from collections import defaultdict
from requests import HTTPError
from requests.adapters import HTTPAdapter
from urllib import urlencode
from urllib3.util import Retry


SECRETS_FILE = '/etc/qnductor-sec.json'
BASIC_LOG = '/var/log/qnductor.log'
STATUS_FILE = '/var/log/qnductor.status'
logging.basicConfig(filename=BASIC_LOG,
                    level=logging.INFO,
                    format="%(asctime)s: %(levelname)s: %(funcName)s %(message)s",
                    datefmt="%m/%d/%Y %H:%M:%S ")

logger = logging.getLogger('qnductor')

CONDUCTOR_URL = "https://c.yandex-team.ru"
CONDUCTOR_API = CONDUCTOR_URL + "/api/v1/"
CONDUCTOR_API_HANDLES = CONDUCTOR_URL + "/api/"
QLOUD_API = 'https://qloud-ext.yandex-team.ru/api/v1'
QLOUD_API_TIMEOUT = 60
QLOUD_API_RETRIES = 3

conductor_project_id = 35
conductor_main_parent_group_id = 26899

QLOUD_PARENT_GROUP = "qloud_mail"
QLOUD_APPLICATIONS_BLACKLIST = ["mail.wmi-qa", "mail.verstka-qa", "mail.maya-qa", "mail.sarah-qa", "mail.payments-qa", "mail.mailfront"]
QLOUD_COMPONENT_TYPES_EXCLUDED = ["component-proxy", "proxy"]

secrets = json.load(open(SECRETS_FILE))

QLOUD_TOKEN = secrets['QLOUD_TOKEN']
QLOUD_AUTH_HEADERS = {"Authorization": "OAuth " + QLOUD_TOKEN}

CONDUCTOR_TOKEN = secrets['CONDUCTOR_TOKEN']
CONDUCTOR_AUTH_HEADERS = {"Authorization": "OAuth " + CONDUCTOR_TOKEN}
CONDUCTOR_API_TIMEOUT = 30

NON_DEPLOYED_ENVIRONMENTS = []
FORCE_CLEAN_GROUPS = []

class Conductor:
    def __init__(self):
        pass
        self._CONDUCTOR_GROUPS = {}

    def create_group(self, name, data):
        res = self._post('groups', data)
        logger.info("%s: %s" % (name, res.status_code))

    def add_host_to_group(self, host, data):
        res = self._post('hosts', data)
        logger.info("%s: %s" % (host, res.status_code))

    def delete_host(self, host):
        res = self._delete('hosts/' + host)
        logger.info("%s: %s" % (host, res.status_code))

    def get_groups_with_hosts(self):
        if not self._CONDUCTOR_GROUPS:
            lines = self._get_hanldes_api('/groups_export?format=json').text.splitlines()
            for line in lines:
                group = line.split(':')[0]
                hosts = line.split(':')[1].split(',')
                if hosts[0] == "":
                    hosts = []
                self._CONDUCTOR_GROUPS[group] = hosts
        return self._CONDUCTOR_GROUPS

    def _post(self, endpoint, data):
        res = requests.post(CONDUCTOR_API + endpoint, headers=CONDUCTOR_AUTH_HEADERS, data=data, timeout=CONDUCTOR_API_TIMEOUT)
        res.raise_for_status()
        return res

    def _delete(self, endpoint):
        res = requests.delete(CONDUCTOR_API + endpoint, headers=CONDUCTOR_AUTH_HEADERS, timeout=CONDUCTOR_API_TIMEOUT)
        res.raise_for_status()
        return res

    def _get_hanldes_api(self, endpoint):
        res = requests.get(CONDUCTOR_API_HANDLES + endpoint, headers=CONDUCTOR_AUTH_HEADERS, timeout=CONDUCTOR_API_TIMEOUT)
        res.raise_for_status()
        return res

CONDUCTOR = Conductor()


class Qloud:
    def __init__(self):
        self._session = self._make_keepalive_session()

    def load_project_info(self, project):
        res = self._make_get_request(QLOUD_API + "/project/" + project)
        return res.json()

    def load_environment_info(self, env):
        res = self._make_get_request(QLOUD_API + "/environment/stable/" + env)
        return res.json()

    def _make_keepalive_session(self, retries=QLOUD_API_RETRIES, status_forcelist=(499, 500, 502, 503, 504)):
        session = requests.Session()
        retry = Retry(total=retries, read=retries, connect=retries, backoff_factor=0.3, status_forcelist=status_forcelist)
        adapter = HTTPAdapter(max_retries=retry)
        session.mount('http://', adapter)
        session.mount('https://', adapter)
        return session

    def _make_get_request(self, url):
        res = self._session.get(url, headers=QLOUD_AUTH_HEADERS, timeout=QLOUD_API_TIMEOUT)
        res.raise_for_status()
        return res

QLOUD = Qloud()


def load_qnductor_status():
    try:
        with open(STATUS_FILE) as f:
            return json.load(f)
    except:
        return {}


def  save_qnductor_status(status):
    with open(STATUS_FILE, 'w') as f:
        json.dump(status, f)


def get_project_environments(qloud_project):
    project_environments = []

    contents = QLOUD.load_project_info(qloud_project)
    if contents:
        logger.info("ok")
    for application in contents["applications"]:
        if application["objectId"] in QLOUD_APPLICATIONS_BLACKLIST:
            continue
        for environment in application["environments"]:
            if "status" not in environment or environment["status"] != "DEPLOYED":
                NON_DEPLOYED_ENVIRONMENTS.append(environment["objectId"])
            if environment.get("upstreamComponents"):
                continue
            project_environments.append(environment["objectId"])
    return project_environments


def get_environment_components(qloud_environment):
    components = defaultdict(list)
    contents = QLOUD.load_environment_info(qloud_environment)
    if contents:
        logger.info("ok")
    for component in contents["components"].values():
        component_name = component["objectId"]
        if component['type'] in QLOUD_COMPONENT_TYPES_EXCLUDED:
            FORCE_CLEAN_GROUPS.append(escape_conductor_group(component_name))
            components[component_name] = []
            continue
        for instance in component["runningInstances"]:
            # instance может не именть "host":
            if instance.get("host"):
                components[component_name].append(instance["host"])
    return components


def get_conductor_group_names():
    return CONDUCTOR.get_groups_with_hosts().keys()


def get_conductor_group_hosts(name):
    if name in CONDUCTOR.get_groups_with_hosts():
        return CONDUCTOR.get_groups_with_hosts()[name]
    return []


def escape_conductor_group(name):
    return re.sub('\.', '_', name)


def create_conductor_group(name):
    logger.info(name)
    host_data = {'group[name]': name,
                    'group[project_id]': conductor_project_id,
                    'group[description]': name,
                    'group[parent_ids][]': conductor_main_parent_group_id,
                    'group[export_to_racktables]': 'false'}
    host_data = urlencode(host_data)
    CONDUCTOR.create_group(name, host_data)


def delete_hosts_from_conductor_group(group, hosts):
    logger.info(group)
    for host in hosts:
        logger.info("host: {}".format(host))
        CONDUCTOR.delete_host(host)


def add_hosts_to_conductor_group(group, hosts):
    logger.info(group)
    for host in hosts:
        try:
            logger.info("host: {}".format(host))
        except UnicodeEncodeError:
            logger.error("'ascii' codec can't encode characters, skip host")
            continue

        if re.search(" ", host):
            logger.error("Bad host format in group: {}".format(group))
            continue
        short = re.sub('\..*', '', host)
        dc = re.sub('-.*', '', host)
        dc = re.sub('sas.*', 'sas', dc)
        dc = re.sub('vla.*', 'vla', dc)
        host_data = {'host[fqdn]': host,
                        'host[datacenter_name]': dc,
                        'host[short_name]': short,
                        'host[group_name]': group}
        host_data = urlencode(host_data)
        CONDUCTOR.add_host_to_group(host, host_data)


def update_conductor_hosts(group, target_hosts):
    current_hosts = get_conductor_group_hosts(group)

    hosts_to_del = set(current_hosts) - set(target_hosts)
    hosts_to_add = set(target_hosts) - set(current_hosts)

    if len(hosts_to_del) > 0:
        # Проверяем, что чистим не мета-группу:
        if len(target_hosts) == 0 and group not in FORCE_CLEAN_GROUPS:
            logger.info("skipping purging empty group")
            return
        delete_hosts_from_conductor_group(group, hosts_to_del)

    if len(hosts_to_add) > 0:
        add_hosts_to_conductor_group(group, hosts_to_add)


def process_environment(env):
    logger.info(env)

    if env in NON_DEPLOYED_ENVIRONMENTS:
        logger.info("skipping non-deployed environment")
        return

    components = get_environment_components(env)
    exception_occured = False
    for name, hosts in components.items():
        # try to process every component, even if some of them failing
        try:
            name = escape_conductor_group(name)
            if name not in get_conductor_group_names():
                create_conductor_group(name)
            update_conductor_hosts(name, hosts)
        except HTTPError as x:
            logger.error("HTTPError: {}".format(x.response.content))
            exception_occured = True
        except Exception as x:
            logging.exception(str(x))
            exception_occured = True

    if exception_occured:
        raise RuntimeError("environment not fully processed")



def main(args):
    status = load_qnductor_status()
    try:
        if args.environment:
            project_environments = [args.environment]
        else:
            project_environments = get_project_environments("mail")
    except HTTPError as x:
        logger.error("HTTPError: {}".format(x.response.content))
    except Exception as x:
        logging.exception(str(x))
        exit(1)

    update_status_ts = True
    for environment in project_environments:
        try:
            process_environment(environment)
        except HTTPError as x:
            logger.error("HTTPError: {}".format(x.response.content))
            update_status_ts = False
        except Exception as x:
            logging.exception(str(x))
            update_status_ts = False
    if update_status_ts:
        status['environment_update_ts'] = int(time.time())

    save_qnductor_status(status)

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("-e", "--environment", type=str, default=None,
                        required=False, help="Update specific qloud environment")
    args = parser.parse_args()
    main(args)
