#!/usr/bin/env python3

import sys
import shutil
import click
import logging
import requests
import re
import json
import base64
import urllib3

from typing import List, Set, Mapping, OrderedDict, Optional, Callable
from pathlib import Path, PurePath
from dataclasses import dataclass, field
from urllib.parse import quote
from datetime import datetime


def setup_logger(level: str = 'WARNING'):
    handler = logging.StreamHandler(sys.stdout)
    handler.setFormatter(logging.Formatter('[{asctime}] [{levelname:^8}]    {name:<8} - {message}', style='{'))

    logger = logging.getLogger()
    logger.addHandler(handler)
    logger.setLevel(level)


@dataclass
class Envelope:
    Mid: str
    Info: Mapping
    DataFile: Optional[str] = None

    def dump(self, outputFolderPath: str) -> None:
        Path(outputFolderPath).mkdir(parents=True, exist_ok=True)

        with Path(outputFolderPath, '{}.json'.format(self.Mid)).open('w') as envelopeFile:
            json.dump(self.Info, envelopeFile, indent=4)

    @staticmethod
    def dumpAll(envelopes: Mapping[str, 'Envelope'], outputFolderPath: str) -> None:
        for _, envelope in envelopes.items():
            envelope.dump(outputFolderPath)

    @staticmethod
    def load(inputEnvelopePath: str) -> Optional['Envelope']:
        path = Path(inputEnvelopePath)
        if not (path.is_file() and re.match(r'\d+\.json', path.name)):
            return None

        with path.open() as envelopeFile:
            info = json.load(envelopeFile)

        mid = re.match(r'(?P<mid>\d+)\.json', path.name).group('mid')
        dataFile = path.parent.joinpath('{}.eml'.format(mid))

        return Envelope(
            Mid=mid,
            Info=info,
            DataFile=str(dataFile) if dataFile.is_file() else None
        )

    @staticmethod
    def loadAll(inputFolderPath: str) -> Mapping[str, 'Envelope']:
        def filterEnvelope(x: Path) -> bool:
            return x.is_file() and re.match(r'\d+\.json', x.name)

        envelopes = OrderedDict()
        for x in filter(filterEnvelope, Path(inputFolderPath).iterdir()):
            envelope = Envelope.load(str(x))
            if envelope:
                envelopes[envelope.Mid] = envelope

        return envelopes


@dataclass
class Folder:
    Name: str
    Info: Mapping
    Folders: Mapping[int, 'Folder'] = field(default_factory=OrderedDict)
    Envelopes: Mapping[str, Envelope] = field(default_factory=OrderedDict)

    def dump(self, outputFolderPath: str) -> None:
        Path(outputFolderPath).mkdir(parents=True, exist_ok=True)

        with Path(outputFolderPath, 'info.json').open('w') as infoFile:
            json.dump(self.Info, infoFile, indent=4)

        Envelope.dumpAll(self.Envelopes, outputFolderPath)

        Folder.dumpAll(self.Folders, outputFolderPath)

    @staticmethod
    def dumpAll(folders: Mapping[int, 'Folder'], outputFolderPath: str) -> None:
        for fid, folder in folders.items():
            folder.dump(str(PurePath(outputFolderPath, str(fid))))

    @staticmethod
    def load(inputFolderPath: str) -> Optional['Folder']:
        path = Path(inputFolderPath)
        if not (path.is_dir() and re.match(r'\d+', path.name)):
            return None

        infoPath = path.joinpath('info.json')
        if not infoPath.is_file():
            return None

        with infoPath.open() as infoFile:
            info = json.load(infoFile)

        return Folder(
            Name=info['name'],
            Info=info,
            Folders=Folder.loadAll(inputFolderPath),
            Envelopes=Envelope.loadAll(inputFolderPath)
        )

    @staticmethod
    def loadAll(inputFolderPath: str) -> Mapping[int, 'Folder']:
        def filterFolder(x: Path) -> bool:
            return x.is_dir() and re.match(r'\d+', x.name)

        folders = OrderedDict()
        for x in filter(filterFolder, Path(inputFolderPath).iterdir()):
            folder = Folder.load(str(x))
            if folder:
                folders[int(folder.Info['id'])] = folder

        return folders


@dataclass
class MailBox:
    Folders: Mapping[int, Folder] = field(default_factory=OrderedDict)

    def folderList(self) -> List[List[Folder]]:
        def listFolders(prefix: List[Folder], folders: Mapping[int, Folder]) -> List[List[Folder]]:
            result = []
            for folder in folders.values():
                newPrefix = prefix.copy() + [folder]
                result.append(newPrefix)

                result += listFolders(prefix=newPrefix, folders=folder.Folders)

            return result

        return listFolders(prefix=[], folders=self.Folders)

    def dump(self, outputFolderPath: str) -> None:
        Path(outputFolderPath).mkdir(parents=True, exist_ok=True)

        Folder.dumpAll(self.Folders, outputFolderPath)

    @staticmethod
    def load(inputFolderPath: str) -> 'MailBox':
        return MailBox(Folder.loadAll(inputFolderPath))


class ServiceBase():
    def __init__(self, name: str, host: str, port: Optional[int]):
        self.Name = name
        self.Host = host
        self.Port = port

        self.Protocol = 'https'
        self.ServiceTicket = None

        self.logger = logging.getLogger(self.Name)
        self.logger.setLevel('INFO')

    def extra_headers(self) -> Mapping[str, str]:
        return {'X-Ya-Service-Ticket': self.ServiceTicket} if self.ServiceTicket else {}

    def make_url(self, path, protocol=None, **kwargs) -> str:
        protocol = protocol if protocol is not None else self.Protocol
        return '{protocol}{host}{port}{path}{args}'.format(
            protocol='{}://'.format(protocol) if protocol else '',
            host=self.Host,
            port=':{}'.format(self.Port) if self.Port else '',
            path=path,
            args='?{}'.format('&'.join(['{}={}'.format(k, quote(str(v))) for k, v in kwargs.items()])) if kwargs else ''
        )


class Hound(ServiceBase):
    def __init__(self, host: str, port: Optional[int], ticket: Optional[str] = None):
        super(Hound, self).__init__(
            name="hound",
            host=host,
            port=port
        )

        self.Protocol = 'https'
        self.ServiceTicket = ticket

    def readMailBox(self, uid: str, onlyFolders: bool = False, folderFilter: Callable[[Mapping], bool] = lambda x: True) -> MailBox:
        mailBox = self._folders(uid=uid, folderFilter=folderFilter)
        if not onlyFolders:
            self._messages_by_folder(uid=uid, folders=mailBox.Folders)

        return mailBox

    def _folders(self, uid: str, folderFilter: Callable[[Mapping], bool]) -> MailBox:
        url = self.make_url('/v2/folders', uid=uid)
        headers = self.extra_headers()
        self.logger.info('folders: uid={uid}'.format(uid=uid))
        resp = requests.get(url, headers=headers, timeout=30, verify=False)
        if resp.status_code != 200:
            self.logger.error('folders: status_code={status}, text={text}'.format(status=resp.status_code, text=resp.text))
            return None

        folders = json.loads(resp.text)["folders"]

        folder_list = OrderedDict([(int(x['id']), Folder(Name=x['name'], Info=x)) for x in folders])

        def acceptFid(fid: int, fids: Set[int]) -> None:
            if 0 < fid and fid not in fids:
                fids.add(fid)

                parentId = int(folder_list[fid].Info['parentId'])
                acceptFid(parentId, fids)

        fids = set()
        for fid, folder in folder_list.items():
            if folderFilter(folder.Info):
                acceptFid(fid, fids)

        mailBox = MailBox()
        for fid, folder in folder_list.items():
            if fid not in fids:
                continue

            parentId = int(folder.Info['parentId'])
            foldersOfParent = folder_list[parentId].Folders if parentId in folder_list else mailBox.Folders

            foldersOfParent[fid] = folder

        return mailBox

    def _messages_by_folder(self, uid: str, folders: Mapping[int, Folder]) -> None:
        headers = self.extra_headers()
        count = 100

        for fid, folder in folders.items():
            self.logger.info('messages_by_folder: fid={fid}'.format(fid=fid))

            envelopes = []
            while True:
                url = self.make_url('/messages_by_folder', uid=uid, fid=fid, first=len(envelopes), count=count)
                resp = requests.get(url, headers=headers, timeout=30, verify=False)
                envelopesChunk = json.loads(resp.text)["envelopes"]

                if not envelopesChunk:
                    break

                envelopes += envelopesChunk

            folder.Envelopes = OrderedDict([(x['mid'], Envelope(x['mid'], x)) for x in envelopes])

            self._messages_by_folder(uid=uid, folders=folder.Folders)


class Mbody(ServiceBase):
    def __init__(self, host: str, port: Optional[int], ticket: Optional[str] = None):
        super(Mbody, self).__init__(
            name="mbody",
            host=host,
            port=port
        )

        self.Protocol = 'http'
        self.ServiceTicket = ticket

    def readMessages(self, uid: str, outDir: str, mailBox: MailBox, folderFilter: Callable[[Mapping], bool] = lambda x: True) -> None:
        self._read_message_sources(uid, outDir, mailBox.Folders, folderFilter)

    def _read_message_sources(self, uid: str, outputDirectory: str, folders: Mapping[int, Folder], folderFilter: Callable[[Mapping], bool]) -> None:
        path = Path(outputDirectory)
        for fid, folder in folders.items():
            if not folderFilter(folder.Info):
                continue

            self._read_folder_message_sources(uid, str(path.joinpath(str(fid))), folder, folderFilter)

    def _read_folder_message_sources(self, uid: str, outputDirectory: str, folder: Folder, folderFilter: Callable[[Mapping], bool]) -> None:
        path = Path(outputDirectory)
        path.mkdir(parents=True, exist_ok=True)
        for mid in folder.Envelopes:
            messageFilePath = path.joinpath('{}.eml'.format(mid))
            if messageFilePath.exists():
                continue

            data = self._message_source(uid=uid, mid=mid)
            if data:
                with messageFilePath.open('wb') as messageFile:
                    messageFile.write(data)

        self._read_message_sources(uid, outputDirectory, folder.Folders, folderFilter)

    def _message_source(self, uid: str, mid: str) -> Optional[bytes]:
        url = self.make_url('/message_source', uid=uid, mid=mid)
        headers = self.extra_headers()
        self.logger.info('message_source: reading message with mid={mid}'.format(mid=mid))
        resp = requests.get(url, headers=headers, timeout=600, verify=False)
        if resp.status_code != 200:
            self.logger.error('message_source: status_code={status}, text={text}'.format(status=resp.status_code, text=resp.text))
            return None

        return base64.b64decode(json.loads(resp.text)['text'])


class Mops(ServiceBase):
    def __init__(self, host: str, port: Optional[int], ticket: Optional[str] = None):
        super(Mops, self).__init__(
            name="mops",
            host=host,
            port=port
        )

        self.Protocol = 'http'
        self.ServiceTicket = ticket

    def generateFoldersTree(self, uid: str, expected: Mapping[int, Folder], existing: Mapping[int, Folder], folderFilter: Callable[[Mapping], bool]) -> Mapping[int, Optional[int]]:
        return self._generate_folders_tree(uid, expected, existing, parentFid=0, folderFilter=folderFilter)

    def _generate_folders_tree(self, uid: str, expected: Mapping[int, Folder], existing: Mapping[int, Folder], parentFid: int, folderFilter: Callable[[Mapping], bool]) -> Mapping[int, Optional[int]]:
        res: Mapping[int, int] = {}
        for fid, folder in expected.items():
            res[fid] = None
            if not folderFilter(folder.Info):
                continue

            for existingFid, existingFolder in existing.items():
                if existingFolder.Name == folder.Name:
                    res[fid] = existingFid
                    res.update(self._generate_folders_tree(uid, folder.Folders, existingFolder.Folders, existingFid, folderFilter))
                    break
            else:
                if folder.Info['type'] == 'user':
                    createdFid = self._folders_create(uid, folder.Name, parentFid)
                    if createdFid is not None:
                        res[fid] = createdFid
                        res.update(self._generate_folders_tree(uid, folder.Folders, {}, createdFid, folderFilter))
                    else:
                        self.logger.error('generate_folders_tree: failed to create folder {name}'.format(name=folder.Name))

        return res

    def _folders_create(self, uid: str, folderName: str, parentFid: Optional[int] = None) -> Optional[int]:
        url = self.make_url('/folders/create', uid=uid, name=folderName)
        headers = self.extra_headers()
        data = [('name', folderName)] + [('parent_fid', parentFid)] if parentFid else []
        self.logger.info('folders_create: creating folder name={name}, parentFid={fid}'.format(name=folderName, fid=parentFid))
        resp = requests.post(url, headers=headers, data=data, timeout=30, verify=False)
        if resp.status_code != 200:
            self.logger.error('folders_create: status_code={status}, text={text}'.format(status=resp.status_code, text=resp.text))
            return None

        return int(json.loads(resp.text)['fid'])


class MxBack(ServiceBase):
    def __init__(self, host: str, port: Optional[int], ticket: Optional[str] = None):
        super(MxBack, self).__init__(
            name="mxback",
            host=host,
            port=port
        )

        self.Protocol = 'https'
        self.ServiceTicket = ticket

    def saveAll(self, email: str, folders: Mapping[int, Folder], folderMapping=Mapping[int, Optional[int]], removeAfterSave: bool = False, folderFilter: Callable[[Mapping], bool] = lambda x: True):
        def _save_all(folders: Mapping[int, Folder]) -> None:
            for fid, folder in folders.items():
                if not folderFilter(folder.Info):
                    continue

                targetFid = folderMapping.get(fid)
                if targetFid is None:
                    continue

                for envelope in folder.Envelopes.values():
                    if envelope.DataFile:
                        timestamp = datetime.fromtimestamp(envelope.Info['date'])
                        if self._save(datafile=envelope.DataFile, email=email, fid=targetFid, timestamp=timestamp) and removeAfterSave:
                            Path(envelope.DataFile).unlink()

                _save_all(folder.Folders)

        _save_all(folders)

    def _save(self, datafile: str, email: str, fid: int, timestamp: datetime = datetime.now()) -> bool:
        url = self.make_url('/save', src_email=email, fid=fid, request_id='stamp-{}'.format(int(timestamp.timestamp())))
        headers = self.extra_headers()
        with open(datafile, 'rb') as file:
            data = file.read()
            self.logger.info('save: file={file}, fid={fid}, size={size}kB'.format(file=datafile, fid=fid, size='{:.1f}'.format(len(data)/1024)))
            resp = requests.post(url, headers=headers, data=data, timeout=600, verify=False)
            if resp.status_code != 200:
                self.logger.error('save: status_code={status}, text={text}, file={file}'.format(status=resp.status_code, text=resp.text, file=datafile))
                return False

            return True


@click.group()
def cli():
    urllib3.disable_warnings()
    setup_logger(level='INFO')


@cli.command('read-mail-box')
@click.option('--uid', required=True, help='User ID')
@click.option('--exclude', help='Folder ID to exclude', multiple=True)
@click.option('--dir', required=True, help='Directory to store/load mail-box data')
@click.option('--clean', 'clear_directory', is_flag=True, help='Clean directory')
@click.option('--hound-host', 'hound_host', help='Hound (meta) host')
@click.option('--hound-port', 'hound_port', type=int, help='Hound (meta) port')
@click.option('--hound-ticket', 'hound_ticket', help='Hound (meta) TVM2 service ticket')
@click.option('--mbody-host', 'mbody_host', help='MBody host')
@click.option('--mbody-port', 'mbody_port', type=int, help='MBody port')
@click.option('--mbody-ticket', 'mbody_ticket', help='MBody TVM2 service ticket')
def read_mail_box(
    uid,
    exclude,
    dir,
    clear_directory,
    hound_host,
    hound_port,
    hound_ticket,
    mbody_host,
    mbody_port,
    mbody_ticket,
):

    logging.info('read_mail_box: started')

    def folderFilter(folderInfo: Mapping) -> bool:
        return int(folderInfo['id']) not in exclude

    if hound_host:
        hound = Hound(hound_host, hound_port, hound_ticket)
        mailBox = hound.readMailBox(uid=uid, folderFilter=folderFilter)

        if clear_directory:
            shutil.rmtree(dir, ignore_errors=True)

        mailBox.dump(dir)
    else:
        mailBox = MailBox.load(dir)

    if not mailBox:
        logging.error('read_mail_box: mailbox not available')
        return

    if mbody_host:
        mbody = Mbody(mbody_host, mbody_port, mbody_ticket)
        mbody.readMessages(uid=uid, outDir=dir, mailBox=mailBox, folderFilter=folderFilter)

    logging.info('read_mail_box: finished')


@cli.command('write-mail-box')
@click.option('--uid', required=True, help='User ID')
@click.option('--email', required=True, help='User E-mail')
@click.option('--exclude', help='Folder ID to exclude', multiple=True)
@click.option('--dir', required=True, help='Directory to load mail-box data from')
@click.option('--remove-saved', 'remove_after_save', is_flag=True, help='Remove source *.eml file after save')
@click.option('--hound-host', 'hound_host', help='Hound (meta) host')
@click.option('--hound-port', 'hound_port', type=int, help='Hound (meta) port')
@click.option('--hound-ticket', 'hound_ticket', help='Hound (meta) TVM2 service ticket')
@click.option('--mops-host', 'mops_host', help='Mops host')
@click.option('--mops-port', 'mops_port', type=int, help='Mops port')
@click.option('--mops-ticket', 'mops_ticket', help='Mops TVM2 service ticket')
@click.option('--mxback-host', 'mxback_host', help='MxBack host')
@click.option('--mxback-port', 'mxback_port', type=int, help='MxBack port')
@click.option('--mxback-ticket', 'mxback_ticket', help='MxBack TVM2 service ticket')
def write_mail_box(
    uid,
    email,
    exclude,
    dir,
    remove_after_save,
    hound_host,
    hound_port,
    hound_ticket,
    mops_host,
    mops_port,
    mops_ticket,
    mxback_host,
    mxback_port,
    mxback_ticket,
):

    logging.info('write_mail_box: started')

    def folderFilter(folderInfo: Mapping) -> bool:
        return int(folderInfo['id']) not in exclude

    srcMailBox = MailBox.load(dir)
    if not srcMailBox:
        return

    hound = Hound(hound_host, hound_port, hound_ticket)
    dstMailBox = hound.readMailBox(uid=uid, onlyFolders=True)

    mops = Mops(mops_host, mops_port, mops_ticket)
    folderMapping = mops.generateFoldersTree(uid, srcMailBox.Folders, dstMailBox.Folders, folderFilter)

    mxback = MxBack(mxback_host, mxback_port, mxback_ticket)
    mxback.saveAll(email=email, folders=srcMailBox.Folders, folderMapping=folderMapping, folderFilter=folderFilter, removeAfterSave=remove_after_save)

    logging.info('write_mail_box: finished')


if __name__ == '__main__':
    cli()
