import click
import psycopg2

from mail.tools.dbaas.helpers.types.env import Envs, Env
from mail.tools.dbaas.helpers.yav import get_users_from_yav
from mail.tools.dbaas.helpers.shards import get_shard_id, get_users, get_shard_users, report_users, UserPresence


def get_shard_tables(shard_conn, schema):
    cur = shard_conn.cursor()
    cur.execute(
        '''
        SELECT table_name
          FROM information_schema.tables
          WHERE table_schema = %(schema_name)s and table_name not like %(filter)s
        ''',
        vars=dict(
            schema_name=schema,
            filter='%_log%',
        )
    )
    return [row[0] for row in cur.fetchall()]


def get_table_records_count(shard_conn, schema, table):
    cur = shard_conn.cursor()
    cur.execute(
        '''
        SELECT n_live_tup
          FROM pg_stat_user_tables
          WHERE relname = %(table_name)s and schemaname = %(schema_name)s
        ''',
        vars=dict(
            schema_name=schema,
            table_name=table,
        )
    )
    res = cur.fetchone()
    return res[0] if res else 0


def print_schema_records_count(shard_conn, schema):
    for table in get_shard_tables(shard_conn, schema):
        records_cnt = get_table_records_count(shard_conn, schema, table)
        if records_cnt:
            print(f'{schema}.{table} has {records_cnt} records')


def print_shard_records_count(shard_dsn):
    print('*** Check records count in DB')
    with psycopg2.connect(shard_dsn) as conn:
        print_schema_records_count(conn, 'mail')
        print_schema_records_count(conn, 'mailish')
        print_schema_records_count(conn, 'ora')
        print_schema_records_count(conn, 'filters')
        print_schema_records_count(conn, 'contacts')
        print_schema_records_count(conn, 'settings')


def get_alive_user_shard_id(conn, uid):
    cur = conn.cursor()
    cur.execute(
        'select shard_id from shards.users where uid = %(uid)s',
        vars=dict(
            uid=uid,
        )
    )
    shard_id = cur.fetchone()
    if shard_id is not None:
        shard_id = shard_id[0]
    return shard_id


def get_deleted_user_shard_id(conn, uid):
    cur = conn.cursor()
    cur.execute(
        'select shard_id from shards.deleted_users where uid = %(uid)s',
        vars=dict(
            uid=uid,
        )
    )
    shard_id = cur.fetchone()
    if shard_id is not None:
        shard_id = shard_id[0]
    return shard_id


def get_user(dsn, uid):
    with psycopg2.connect(dsn) as conn:
        cur = conn.cursor()
        cur.execute(
            'select is_here, is_deleted from mail.users where uid = %(uid)s',
            vars=dict(
                uid=uid,
            )
        )
        row = cur.fetchone()
        if row is not None:
            return row[0], row[1]
        return None, None


def check_users(shard_name, shard_users, alive_users, deleted_users):
    print('*** Check users')
    report_users(
        shard_name,
        alive_users - shard_users[UserPresence(is_here=True, is_deleted=False)],
        'extra_alive_in_sharddb',
    )
    report_users(
        shard_name,
        deleted_users - shard_users[UserPresence(is_here=True, is_deleted=True)],
        'extra_deleted_in_sharddb',
    )

    extra_alive_in_maildb = shard_users[UserPresence(is_here=True, is_deleted=False)] - alive_users
    report_users(
        shard_name,
        extra_alive_in_maildb,
        'extra_alive_in_maildb'
    )

    extra_deleted_in_maildb = shard_users[UserPresence(is_here=True, is_deleted=True)] - deleted_users
    report_users(
        shard_name,
        extra_deleted_in_maildb,
        'extra_deleted_in_maildb'
    )

    non_transfered_alive_in_maildb = shard_users[UserPresence(is_here=True, is_deleted=False)] & alive_users
    report_users(
        shard_name,
        non_transfered_alive_in_maildb,
        'non_transfered_alive_in_maildb'
    )
    if (len(non_transfered_alive_in_maildb) > 0):
        print("!!!!! AHTUNG. There are some non transfered alive users in shard")

    non_transfered_deleted_in_maildb = shard_users[UserPresence(is_here=True, is_deleted=True)] & deleted_users
    report_users(
        shard_name,
        non_transfered_deleted_in_maildb,
        'non_transfered_deleted_in_maildb'
    )
    if (len(non_transfered_deleted_in_maildb) > 0):
        print("!!!!! AHTUNG. There are some non transfered deleted users in shard")


@click.command('check-shard-clearness')
@click.option('--shard-name', required=True)
@click.option('--env', 'env_name', default=Envs.prod.value.name)
@click.option('--db_user', default='transfer')
def main(shard_name: str, env_name: str, db_user: str):
    print(f'************************** Begin processing shard: {shard_name}')

    env: Env = Envs[env_name].value
    users = get_users_from_yav(env.users_file)

    shard_id = get_shard_id(env.sharpei_dsn, shard_name)
    print(f'Shard id: {shard_id}')

    with psycopg2.connect(env.sharpei_dsn) as sharpei_conn:
        shard_dsn = get_shard_master_dsn(sharpei_conn, shard_id, db_user, users[db_user]['password'])

        print_shard_records_count(shard_dsn)

        alive_users, deleted_users = get_users(env.sharpei_dsn, shard_id)
        shard_users = get_shard_users(shard_dsn)

        check_users(shard_name, shard_users, alive_users, deleted_users)

        print("*** Check extra users")
        extra_alive_in_maildb = shard_users[UserPresence(is_here=True, is_deleted=False)] - alive_users
        extra_deleted_in_maildb = shard_users[UserPresence(is_here=True, is_deleted=True)] - deleted_users
        extra_users = extra_alive_in_maildb | extra_deleted_in_maildb
        for uid in extra_users:
            new_shard_id = get_alive_user_shard_id(sharpei_conn, uid)
            # is_deleted_in_sharpei = False
            if new_shard_id is None:
                new_shard_id = get_deleted_user_shard_id(sharpei_conn, uid)
                # is_deleted_in_sharpei = True
            if new_shard_id is None:
                print(f'!!!!! AHTUNG. Absent user {uid} in sharddb')
                continue
            print(f'user {uid} really in shard {new_shard_id} in sharpei')
            new_shard_dsn = get_shard_master_dsn(sharpei_conn, new_shard_id, db_user, users[db_user]['password'])
            is_here, is_deleted = get_user(new_shard_dsn, uid)
            print(f'   user {uid} in shard with is_here {is_here}, is_deleted {is_deleted}')


def get_shard_master_dsn(conn, shard_id: int, user: str, passwd: str):
    cur = conn.cursor()
    cur.execute(
        'select host from shards.instances where shard_id = %(shard_id)s',
        vars=dict(shard_id=shard_id),
    )
    hosts = [row[0] for row in cur.fetchall()]

    return ' '.join((
        f'host={",".join(hosts)}',
        'port=6432',
        'dbname=maildb',
        f'user={user}',
        f'password={passwd}',
        'target_session_attrs=read-write',
    ))


if __name__ == '__main__':
    main()
