from pprint import pprint

import click
import logging
import sys

from mail.tools.dbaas.helpers.pgmigrate import pgmigrate
from mail.tools.dbaas.helpers.types.env import Envs, Env
from mail.tools.dbaas.helpers.types.user import User
from mail.tools.dbaas.helpers.yc_client import YcClient, PoolingMode, ResourceKinds, ResourceKind
from mail.tools.dbaas.helpers.constants import EXTENSIONS, POSTGRE_CONFIG_OPTS, OWNER_NAME
from mail.tools.dbaas.helpers.yav import get_users_from_yav

from mail.shiva.client.shiva import ShivaClient

from mail.tools.dbaas.helpers.shards import add_shard_to_sharddb, set_shard_workload_exception, SHARD_NAME_RE, MAX_WORKERS_EXCEPTIONS
from mail.tools.dbaas.helpers.shards import get_shard_id, get_shard_master_dsn, get_shard_migration_version

log = logging.getLogger(__name__)


def setup_logger():
    log.setLevel(logging.DEBUG)
    h = logging.StreamHandler(sys.stdout)
    h.setLevel(logging.DEBUG)
    log.addHandler(h)


@click.command('create_cluster')
@click.option('--env', 'env_name', default=Envs.prod.name)
@click.option('--kind', 'resource_kind_name', default=ResourceKinds.common.name)
@click.option('--load-type', default='dbaas_hot')
@click.option('--shard-type', default='general')
@click.option('--name-suffix')
@click.option('--cluster-name')
@click.option('--migrations-dir')
def main(
        env_name: str,
        resource_kind_name: str,
        migrations_dir: str,
        load_type: str,
        shard_type: str,
        name_suffix: str = None,
        cluster_name: str = None
):
    setup_logger()
    env: Env = Envs[env_name].value
    resource_kind: ResourceKind = ResourceKinds[resource_kind_name].value
    users = get_users_from_yav(env.users_file)
    owner_passwd = users[OWNER_NAME]['password']
    yc = YcClient(
        cloud_id=env.cloud_id, folder_name=env.folder_name,
        owner_name=OWNER_NAME, owner_passwd=owner_passwd,
    )
    assert not (name_suffix and cluster_name), 'Only one of `name-suffix` and `cluster-name` can be set'
    if cluster_name is None:
        cluster_name = yc.find_next_cluster_name(env=env.name, suffix=name_suffix)
    users = [
        User(name=uname, **opts)
        for uname, opts in users.items()
    ]
    yc.create_cluster(cluster_name=cluster_name, mdb_env=env.mdb_env_name, users=users, resource_kind=resource_kind)
    for user in users:
        # Grants cannot be given
        if user.grants:
            yc.update_user(cluster_name=cluster_name, user=user)

    yc.create_extensions(cluster_name=cluster_name, extensions=EXTENSIONS)
    yc.update_settings(cluster_name=cluster_name, pooling_mode=PoolingMode.Tx, **POSTGRE_CONFIG_OPTS)
    cluster = next(cluster for cluster in yc.get_clusters() if cluster['name'] == cluster_name)

    hosts = yc.get_hosts(cluster_name)
    print('Hosts:')
    pprint(hosts)
    master_host = next(
        (
            host['name'] for host in hosts
            if host['role'] == 'MASTER'
        ),
        None
    )
    if master_host is None:
        raise RuntimeError('No master found, cannot run migrations')
    latest_migration = pgmigrate(
        basedir=migrations_dir,
        host=master_host,
        user=OWNER_NAME,
        logger=log,
        passwd=owner_passwd,
    )
    shard_name = SHARD_NAME_RE.match(cluster_name).group('shard_name')
    shard_dsn = ' '.join((
        f'host={master_host}',
        'port=6432',
        'dbname=maildb',
        f'user={OWNER_NAME}',
        f'password={owner_passwd}',
        'target_session_attrs=read-write',
    ))
    migration_version = get_shard_migration_version(shard_dsn)

    ref_shard_name = {
        'prod': 'prod-moscow001',
        'corp': 'corp-moscow001',
        'test': 'maildb-test01',
    }[env.name]
    ref_shard_id = get_shard_id(env.sharpei_dsn, ref_shard_name)
    ref_shard_dsn = get_shard_master_dsn(env.sharpei_dsn, ref_shard_id, OWNER_NAME, owner_passwd)
    ref_migration_version = get_shard_migration_version(ref_shard_dsn)

    if str(migration_version) != str(ref_migration_version):
        raise RuntimeError(f'Expected migration version {ref_migration_version}, found version {migration_version}')
    else:
        print(f'Migrations on shard {shard_name} are up-to-date')

    shard_id = add_shard_to_sharddb(
        env=env,
        shard_name=shard_name,
        hosts=hosts,
        load_type=load_type,
    )
    print(f'Shard id: {shard_id}')

    shiva = ShivaClient(
        location=env.shiva_loc,
        logger=log
    )
    shard_json = shiva.admin().shards(shard_id=shard_id).json()
    if str(shard_id) not in shard_json.keys():
        shiva.admin().add_shard(
            shard_id=shard_id,
            cluster_id=cluster['id'],
            shard_type=shard_type,
            migration=latest_migration,
            shard_name=shard_name,
            load_type=load_type,
            can_transfer_to=False,
            disk_size=int(cluster['config']['resources']['disk_size']),
        )
    shard_json = shiva.admin().shards(shard_id=shard_id).json()
    if str(shard_id) in shard_json.keys() and shard_json[str(shard_id)]['cluster_id'] == cluster['id'] and shard_json[str(shard_id)]['shard_name'] == shard_name:
        print(f'Shard {shard_id} was successfully added to shiva')
    else:
        raise RuntimeError(f'Failed to add shard {shard_id} to shiva')

    max_workers_per_shard = MAX_WORKERS_EXCEPTIONS.get(resource_kind.description)
    if max_workers_per_shard is not None:
        set_shard_workload_exception(env, shard_id, max_workers_per_shard)


if __name__ == '__main__':
    main()
