import logging
from typing import Iterable, Callable, Any, Sequence, Tuple
from tqdm import tqdm
from termcolor import colored


def green(msg):
    return colored(msg, 'green')


def red(msg):
    return colored(msg, 'red')


def split(seq: Iterable, pred: Callable[[Any], bool]) -> Tuple[Sequence, Sequence]:
    yes, no = [], []
    for item in seq:
        if pred(item):
            yes.append(item)
        else:
            no.append(item)
    return yes, no


class TqdmLoggingHandler(logging.Handler):
    def __init__(self, level=logging.NOTSET):
        super().__init__(level)

    def emit(self, record):
        try:
            msg = self.format(record)
            tqdm.write(msg)
            self.flush()
        except Exception:
            self.handleError(record)


class TqdmProgressHandler(logging.Handler):
    def __init__(self, bar, level=logging.NOTSET):
        super().__init__(level)
        self.bar = bar

    def emit(self, record):
        if not hasattr(record, 'update_progress'):
            return
        self.bar.update()


def update_progress(log):
    log.info(msg='', extra={'update_progress': True})


class BufferingLogger(object):
    '''
    This class is intended to be used as logger in worker processes.
    It keeps all logged objects in buffer until flush() called explicitly or implicitly during destruction.
    This way the output of every executed task can be printed atomically: it is kept in local buffer until
    the task completion and then logged using the wrapped logger under the interprocess lock.

    Can not be shared between processes or threads, but multiple instances can exist simultaneously in
    different threads/processes.
    '''
    def __init__(self, logger, lock):
        self.logger = logger
        self.lock = lock
        self.buffer = []

    def __del__(self):
        self.flush()

    def info(self, record):
        self._log(record)

    def error(self, record):
        self._log(red(str(record)))

    def exception(self, record):
        self.error(record)

    def _log(self, record):
        self.buffer.append(record)

    def flush(self):
        with self.lock:  # synchronizes access to logger not to buffer, so no locking in BufferingLogger._log()
            for record in self.buffer:
                self.logger.info(record)
        self.buffer = []
