from typing import Iterable, Callable, Any, Sequence

import click

import yaml
from mail.tools.dbaas.helpers.types.env import Envs, Env
from mail.tools.dbaas.helpers.types.user import User
from mail.tools.dbaas.helpers.yc_client import YcClient
from mail.tools.dbaas.helpers.yav import get_users_from_yav

OWNER_NAME = 'maildb'


@click.command('update-user-in-yc')
@click.option('--env', 'env_name', default=Envs.prod.value.name)
@click.option('--user', 'user_name')
def main(env_name: str, user_name: str):
    env: Env = Envs[env_name].value
    users = get_users_from_yav(env.users_file)
    user = User(name=user_name, **users[user_name])
    yc = YcClient(cloud_id=env.cloud_id, folder_name=env.folder_name)
    clusters = yc.get_clusters()

    canary_clusters, main_clusters = split(clusters, pred=lambda cluster: cluster['name_info'].get('suffix') == 'canary')

    if canary_clusters:
        print('Canary clusters found, going to update them first')
        update_user(yc, canary_clusters, user)
        input('Canary migration is finished, press Enter to proceed to main cluster set: ')

    update_user(yc, main_clusters, user)


def split(seq: Iterable, pred: Callable[[Any], bool]) -> (Sequence, Sequence):
    yes, no = [], []
    for item in seq:
        if pred(item):
            yes.append(item)
        else:
            no.append(item)
    return yes, no


def update_user(yc, clusters, user: User):
    print(f'You are going to migrate {len(clusters)} following clusters:')
    print(yaml.dump([
        {k: cluster[k] for k in (
            'name',
            'id',
            'description',
        )}
        for cluster in clusters
    ], indent=4, sort_keys=False))
    input('Press Enter to continue: ')
    for cluster in clusters:
        current_user = None
        current_user_data = next(u for u in yc.get_users(cluster['name']) if u['name'] == user.name)
        if current_user_data:
            current_user = User.from_yc(**current_user_data)
        if current_user != user:
            print('Current user is not equal to proposed settings')
            print(f'Current user: {current_user}')
            print(f'Proposed user: {user}')
            yc.update_user(cluster_name=cluster['name'], user=user)


if __name__ == '__main__':
    main()
