import subprocess
from enum import Enum, unique

import psutil

import json
from mail.tools.dbaas.helpers.types.migration import Migration


@unique
class PgmigrateCmd(Enum):
    Migrate = 'migrate'
    Info = 'info'


def make_pgmigrate_cmd(basedir, dsn, cmd: PgmigrateCmd, target: str = 'latest'):
    return [
        'pgmigrate', cmd.value,
        '-d', str(basedir),
        '-c', dsn,
        '-t', target,
    ]


def run_subprocess(cmd, logger):
    proc = psutil.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True)
    out, err = proc.communicate()
    if out:
        logger.info(out)
    if err:
        logger.error(err)
    return proc.wait()


def pgmigrate(basedir, host, user, logger, passwd=None, one_by_one=False, termination_interval_sec: int = 10):
    '''
    :param logger: whatever supporting methods error(msg), exception(msg), info(msg)
    '''
    logger.info(f'Applying pgmigrate on dir {basedir} for {host}...')
    dsn = f'host={host} port=6432 user={user}'
    if passwd:
        dsn += f' password={passwd}'
    cmd = make_pgmigrate_cmd(basedir, dsn, PgmigrateCmd.Info)
    with psutil.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, text=True) as proc:
        out, err = proc.communicate()
        if err:
            logger.error(err)
        data = json.loads(out)
        migrations = [Migration(**m) for m in data.values()]
    last_current_migration = max([m for m in migrations if m.applied], default=None)
    logger.info(f'Last applied migration: {last_current_migration}')

    if one_by_one:
        unapplied_migrations = sorted([m for m in migrations if not m.applied])
        for migration in unapplied_migrations:
            logger.info(f'Applying migration: {migration}')
            cmd = make_pgmigrate_cmd(basedir, dsn, PgmigrateCmd.Migrate, target=str(migration.version))
            cmd += ['-l', str(termination_interval_sec)]
            logger.info(' '.join(cmd))
            retcode = run_subprocess(cmd, logger)
            if retcode:
                raise RuntimeError(f'pgmigrate run finished with retcode={retcode}')
    else:
        logger.info('Applying all migrations at once')
        cmd = make_pgmigrate_cmd(basedir, dsn, PgmigrateCmd.Migrate)
        cmd += ['-l', str(termination_interval_sec)]
        logger.info(' '.join(cmd))
        retcode = run_subprocess(cmd, logger)
        if retcode:
            raise RuntimeError(f'pgmigrate run finished with retcode={retcode}')
    logger.info(f'Done with {host}')
    return max([m.version for m in migrations], default=0)
