from dataclasses import dataclass, field, fields
from enum import unique, Enum
from typing import List, Sequence, Dict

from mail.tools.dbaas.helpers.constants import DB_NAME


@unique
class UserParamKeyPrefix(Enum):
    ForCreateCluster = ''
    ForCreateUser = '--'


@dataclass
class User:
    name: str
    password: str = field(compare=False, repr=False)
    conn_limit: int
    databases: Sequence[str] = (DB_NAME,)
    grants: Sequence[str] = tuple()
    lock_timeout_ms: int = None
    temp_file_limit: int = 100 * 1024 * 1024  # 100 MB
    log_min_duration_statement: str = None
    log_statement: str = None

    # def __str__(self):
    #     return f'User(name={self.name}, conn_limit={self.conn_limit}, databases={self.databases})'

    def as_items(self):
        for f in fields(self):
            if f.compare:
                yield f.name, getattr(self, f.name)

    @staticmethod
    def log_statement_from_yc(log_statement: str):
        return {
            'LOG_STATEMENT_ALL': 'all'
        }.get(log_statement)

    @classmethod
    def from_yc(
            cls,
            name: str,
            permissions: List[Dict[str, str]],
            conn_limit: str,
            grants: List[str] = tuple(),
            settings: Dict[str, str] = None,
            # Suppress any extra data form YC API
            **kwargs  # noqa
    ):
        lock_timeout_ms = settings.get('lock_timeout') and int(settings.pop('lock_timeout'))
        temp_file_limit = settings.get('temp_file_limit') and int(settings.pop('temp_file_limit'))
        log_statement = cls.log_statement_from_yc(settings.pop('log_statement', None))
        return cls(
            name=name,
            password='XXX',
            conn_limit=int(conn_limit),
            databases=tuple(perm['database_name'] for perm in permissions if 'database_name' in perm),
            grants=grants,
            lock_timeout_ms=lock_timeout_ms,
            temp_file_limit=temp_file_limit,
            log_statement=log_statement,
            **(settings or {}),
        )

    def as_cli_param(self, key_prefix: UserParamKeyPrefix) -> List[str]:
        params = [
            ('password', self.password),
            ('conn-limit', self.conn_limit),
            ('lock-timeout', self.lock_timeout_ms),
            ('temp-file-limit', self.temp_file_limit),
            ('log-min-duration-statement', self.log_min_duration_statement),
            ('log-statement', self.log_statement),
        ]
        if key_prefix == UserParamKeyPrefix.ForCreateCluster:
            params.append(('name', self.name))
            assert len(self.databases) < 2
            params.append(('permission', self.databases and self.databases[0]))
        elif key_prefix == UserParamKeyPrefix.ForCreateUser:
            params.append(('permissions', ','.join(self.databases)))
            params.append(('grants', ','.join(self.grants)))
        return [
            f'{key_prefix.value}{k}={v}' for k, v in params
            if v is not None
        ]
