#!/usr/bin/python3
# -*- coding: UTF-8 -*-

import argparse
import boto
import json
import shutil
import os
import uuid


MAPS_CONFIG_FILE = "/etc/yandex/domain_maps/maps.conf"
S3_SECRET_FILE = "/etc/yandex/domain_maps/s3.secret"
S3_MAPS_BUCKET_NAME = "mail-mx-maps"


def get_maps_bucket(bucket_name, s3_config):
    s3_client = boto.connect_s3(
        aws_access_key_id=s3_config["aws_access_key_id"],
        aws_secret_access_key=s3_config["aws_secret_access_key"],
        host=s3_config["host"]
    )
    return s3_client.get_bucket(bucket_name)


def check_downloaded_size(remote, local):
    if remote != local:
        raise ValueError("Remote size doesn't match local size: {} vs {}".format(remote, local))


def update_domain_map(s3_key, path, apps):
    tmp_path = "/tmp/" + str(uuid.uuid4())

    print("Downloading {} to {}".format(s3_key.name, tmp_path))
    s3_key.get_contents_to_filename(tmp_path)

    check_downloaded_size(s3_key.size, os.path.getsize(tmp_path))

    print("Move {} to {}".format(tmp_path, path))
    shutil.move(tmp_path, path)

    for app in apps:
        print("Reloading {}".format(app))
        if app == "nwsmtp-out":
            if os.path.isfile("/var/run/nwsmtp/nwsmtp-out.pid"):
                os.system("kill -HUP $(cat /var/run/nwsmtp/nwsmtp-out.pid)")
        elif app == "nwsmtp":
            if os.path.isfile("/var/run/nwsmtp/nwsmtp.pid"):
                os.system("kill -HUP $(cat /var/run/nwsmtp/nwsmtp.pid)")
        else:
            os.system("/usr/sbin/service " + app + " reload")


def get_domain_maps(args):
    s3_config = json.load(open(args.s3_secret_file))
    domain_maps = json.load(open(args.maps_config_file))

    maps_bucket = get_maps_bucket(args.s3_maps_bucket_name, s3_config)

    for domain_map, domain_map_options in domain_maps.items():
        path = domain_map_options["path"]
        apps = domain_map_options["apps"]
        s3_key = maps_bucket.get_key(domain_map)
        update_domain_map(s3_key, path, apps)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-c", "--maps-config-file", type=str, default=MAPS_CONFIG_FILE, required=False)
    parser.add_argument("-s", "--s3-secret-file", type=str, default=S3_SECRET_FILE, required=False)
    parser.add_argument("-m", "--s3-maps-bucket-name", type=str, default=S3_MAPS_BUCKET_NAME, required=False)

    args = parser.parse_args()
    get_domain_maps(args)


if __name__ == "__main__":
    main()
