#! /usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import print_function

from dataclasses import dataclass

import requests
import json
import sys


@dataclass(frozen=True)
class UniqueEvent:
    id: str
    date: str


def read_uids(uids_file):
    uids = set()
    with open(uids_file) as f:
        for line in f:
            uid = line.strip()
            uids.add(uid)
    return uids


def is_external_email(email):
    return not "@yandex-team.ru" in email


def is_meeting_with_externals(event):
    for attendee in event['attendees']:
        if is_external_email(attendee['email']):
            return True
    return False


def is_closed(event):
    return event['othersCanView'] is False


def get_user_meetings(uid, date_from, date_to):
    # url = "http://calendar-api.tools.yandex.net/internal/get-events.json"
    url = "http://web-prestable-1.web-prestable.prestable.calendar-yt.mail.stable.qloud-d.yandex.net/internal/get-events.json"
    url += "?uid={}&from={}&to={}".format(uid, date_from, date_to)
    tries = 0
    while True:
        try:
            resp = requests.get(url, timeout=30)
            if resp.status_code != 200:
                raise Exception("calendar response error: {} {}".format(resp.status_code, resp.text))
            break
        except:
            tries += 1
            if tries == 5:
                raise
    return json.loads(resp.text)


def dump_events(events_dir, uids, periods):
    events_per_month = {}
    for date_from, date_to in periods:
        print(f"range={date_from}:{date_to}")
        events = set()
        for uid in uids:
            print(f"uid={uid}")
            user_meetings = get_user_meetings(uid, date_from, date_to).get("events", [])
            print(f"found {len(user_meetings)} meetings")
            for meeting in user_meetings:
                if not is_closed(meeting) and is_meeting_with_externals(meeting):
                    event_datetime = meeting['startTs'].split('T')
                    event_date = event_datetime[0]
                    event = UniqueEvent(id=str(meeting["id"]), date=str(event_date))
                    events.add(event)
        events_per_month[date_from + ':' + date_to] = list(events)

    with open('{}/distribution'.format(events_dir), 'w') as output_file:
        output_file.write(
            '\n'.join([f"{date_range} {len(events)}" for date_range, events in events_per_month.items()])
        )

    print("complete")


def main():
    if len(sys.argv) < 3:
        print("usage: {} <staff uids file> <events dir>".format(sys.argv[0]))
        return 1
    staff_uids_file = sys.argv[1]
    events_dir = sys.argv[2]
    uids = read_uids(staff_uids_file)
    print("read {} uids".format(len(uids)))
    periods = [
        ("2019-01-01", "2019-02-01"),
        ("2019-02-01", "2019-03-01"),
        ("2019-03-01", "2019-04-01"),
        ("2019-04-01", "2019-05-01"),
        ("2019-05-01", "2019-06-01"),
        ("2019-06-01", "2019-07-01"),
        ("2019-07-01", "2019-08-01"),
        ("2019-08-01", "2019-09-01"),
        ("2019-09-01", "2019-10-01"),
        ("2019-10-01", "2019-10-15"),
    ]

    print("will dump for {} uids".format(len(uids)))
    dump_events(events_dir, uids, periods)
    print("done")


if __name__ == '__main__':
    main()
